package ru.yandex.direct.communication.facade.impl.actual;

import java.util.List;
import java.util.function.Function;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.communication.container.AdditionalInfoContainer;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.metrika.client.MetrikaClient;
import ru.yandex.direct.metrika.client.model.response.Counter;
import ru.yandex.direct.rbac.RbacService;

import static ru.yandex.direct.core.security.tvm.TvmUtils.retrieveUserTicket;

@Component
public class MetrikaCounterChecker extends AbstractActualCheckerForClientEvents {

    private final ShardHelper shardHelper;
    private final RbacService rbacService;
    private final MetrikaClient metrikaClient;

    @Autowired
    public MetrikaCounterChecker(
            ShardHelper shardHelper,
            RbacService rbacService,
            MetrikaClient metrikaClient
    ) {
        this.shardHelper = shardHelper;
        this.rbacService = rbacService;
        this.metrikaClient = metrikaClient;
    }

    @Override
    public String getName() {
        return "METRIKA_COUNTER";
    }

    /**
     * Проверяет есть ли у клиента представители с доступом к редактированию каких-либо счетчиков метрики
     * @return true ни один представитель не имеет доступа к редактированию счетчика метрики
     */
    @Override
    protected boolean isEventActualForClient(AdditionalInfoContainer additionalInfo) {
        var clientId = additionalInfo.getClientId().get();
        var representativeUids = List.of(rbacService.getClientRepresentativesUids(clientId));
        List<String> representativeLogins = StreamEx.of(shardHelper.getLoginsByUids(representativeUids))
                .toFlatList(Function.identity());
        var userTicket = retrieveUserTicket();
        return StreamEx.of(metrikaClient.getEditableCounters(userTicket))
                .map(Counter::getOwnerLogin)
                .findAny(representativeLogins::contains)
                .isEmpty();
    }
}
