package ru.yandex.direct.communication.facade.impl.actual;

import java.util.List;
import java.util.Map;
import java.util.Optional;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.communication.container.AdditionalInfoContainer;
import ru.yandex.direct.communication.container.web.SlotMessageId;
import ru.yandex.direct.communication.facade.ActualChecker;
import ru.yandex.direct.communication.facade.impl.actions.ApplyUCGoalsRecommendation;
import ru.yandex.direct.communication.facade.impl.formatter.DefaultFormatter;
import ru.yandex.direct.communication.model.inventory.ObjectEventData;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.DbStrategyBase;
import ru.yandex.direct.core.entity.campaign.model.MeaningfulGoal;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersion;

import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
public class UCGoalsActualChecker implements ActualChecker {

    private final CampaignRepository campaignRepository;

    @Autowired
    public UCGoalsActualChecker(
            CampaignRepository campaignRepository
    ) {
        this.campaignRepository = campaignRepository;
    }

    @Override
    public Map<SlotMessageId, Boolean> check(
            List<ObjectEventData> messagesData,
            Map<SlotMessageId, CommunicationEventVersion> eventVersionByMessageId,
            AdditionalInfoContainer additionalInfo
    ) {
        var shard = additionalInfo.getShard().get();
        var campaignIds = mapList(messagesData, ObjectEventData::getObjectId);
        var meaningfulGoalIdsByCampaignId = EntryStream.of(campaignRepository.getMeaningfulGoalsByCampaignId(shard, campaignIds))
                .nonNullValues()
                .mapValues(goals -> mapList(goals, MeaningfulGoal::getGoalId))
                .toMap();
        var goalByCampaignId = StreamEx.of(campaignRepository.getCampaignsWithStrategy(shard, campaignIds))
                .mapToEntry(
                        Campaign::getId,
                        c -> Optional.ofNullable(c.getStrategy())
                                .map(DbStrategyBase::getStrategyData)
                                .map(StrategyData::getGoalId)
                        .orElse(null)
                )
                .nonNullValues()
                .mapToValue((campId, goalId) -> {
                    if (MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID.equals(goalId)) {
                        return meaningfulGoalIdsByCampaignId.get(campId);
                    } else {
                        return List.of(goalId);
                    }
                })
                .nonNullValues()
                .toMap();
        return StreamEx.of(messagesData)
                .mapToEntry(
                        SlotMessageId::of,
                        md -> {
                            var version = eventVersionByMessageId.get(SlotMessageId.of(md));
                            var data = DefaultFormatter.parseDataFromJson(md.getData(), additionalInfo, version);
                            var goalIds = ApplyUCGoalsRecommendation.getGoalIdsFromData(data);
                            return !goalByCampaignId.getOrDefault(md.getObjectId(), List.of())
                                    .containsAll(goalIds);
                        }
                )
                .toMap();
    }

    @Override
    public String getName() {
        return "UC_GOALS";
    }
}
