package ru.yandex.direct.communication.facade.impl.formatter;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import com.fasterxml.jackson.databind.JsonNode;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.text.StringSubstitutor;
import org.springframework.stereotype.Component;

import ru.yandex.ads.bsyeti.libs.communications.EMessageStatus;
import ru.yandex.ads.bsyeti.libs.communications.ETargetObjectType;
import ru.yandex.ads.bsyeti.libs.communications.EWebUIButtonActionType;
import ru.yandex.ads.bsyeti.libs.communications.EWebUIButtonStyle;
import ru.yandex.ads.bsyeti.libs.communications.EWebUIMessageStatus;
import ru.yandex.ads.bsyeti.libs.communications.proto.TMessageData;
import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.communication.CommunicationHelper;
import ru.yandex.direct.communication.container.AdditionalInfoContainer;
import ru.yandex.direct.communication.container.CommunicationMessageData;
import ru.yandex.direct.communication.container.web.CommunicationDataContainer;
import ru.yandex.direct.communication.container.web.CommunicationLinkInfo;
import ru.yandex.direct.communication.container.web.CommunicationMessage;
import ru.yandex.direct.communication.container.web.CommunicationMessageButton;
import ru.yandex.direct.communication.container.web.CommunicationMessageContent;
import ru.yandex.direct.communication.container.web.CommunicationMessageGroup;
import ru.yandex.direct.communication.container.web.CommunicationMessageInputField;
import ru.yandex.direct.communication.container.web.CommunicationTargetObject;
import ru.yandex.direct.communication.container.web.SlotMessageId;
import ru.yandex.direct.communication.facade.AutoUpdatedSettingsEventFormatter;
import ru.yandex.direct.communication.facade.CommunicationMessageFormatter;
import ru.yandex.direct.communication.facade.CommunicationMessageGroupFormatter;
import ru.yandex.direct.communication.model.Slot;
import ru.yandex.direct.communication.model.inventory.ObjectEventData;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.communication.model.ActionType;
import ru.yandex.direct.core.entity.communication.model.ButtonConfig;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersion;
import ru.yandex.direct.core.entity.communication.model.CommunicationLinkInfoConfig;
import ru.yandex.direct.core.entity.communication.model.Condition;
import ru.yandex.direct.core.entity.communication.model.DataContainer;
import ru.yandex.direct.core.entity.communication.model.FieldConfig;
import ru.yandex.direct.core.entity.communication.model.LinkPath;
import ru.yandex.direct.core.entity.communication.model.Severity;
import ru.yandex.direct.core.entity.communication.model.TextConfig;
import ru.yandex.direct.core.entity.communication.model.ViewType;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.CurrencyTranslation;
import ru.yandex.direct.currency.Percent;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.i18n.Translatable;
import ru.yandex.direct.sender.YandexSenderTemplateParams;
import ru.yandex.direct.useractionlog.model.AutoUpdatedSettingsEvent;
import ru.yandex.direct.utils.JsonUtils;

import static ru.yandex.direct.communication.CommunicationHelper.convertJsonStringToProto;
import static ru.yandex.direct.communication.CommunicationHelper.entityTypeToObjectType;
import static ru.yandex.direct.communication.CommunicationHelper.merge;
import static ru.yandex.direct.communication.CommunicationHelper.transform;
import static ru.yandex.direct.communication.facade.CommunicationEventVersionProcessingFacade.DEFAULT_IMPLEMENTATION;
import static ru.yandex.direct.utils.CommonUtils.nullableNvl;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
public class DefaultFormatter implements CommunicationMessageFormatter, CommunicationMessageGroupFormatter,
        AutoUpdatedSettingsEventFormatter {

    private final TranslationService translationService;

    public DefaultFormatter(TranslationService translationService) {
        this.translationService = translationService;
    }

    @Override
    public List<CommunicationMessageGroup> format(
            List<CommunicationMessage> messages,
            Map<SlotMessageId, CommunicationEventVersion> versionBySlotMessageId,
            AdditionalInfoContainer additionalInfo
    ) {
        var messageById = listToMap(messages, SlotMessageId::of);
        return EntryStream.of(versionBySlotMessageId)
                .invert()
                .mapToKey((version, slotMessageId) -> Pair.of(
                        slotMessageId.getSlotId(),
                        Pair.of(version.getGroupName(), version.getGroupConfig())))
                .mapValues(messageById::get)
                .nonNullValues()
                .collapseKeys()
                .mapKeys(Pair::getRight)
                .mapKeyValue((nameAndConfig, mess) -> {
                    if (nameAndConfig.getRight() == null) {
                        return new CommunicationMessageGroup()
                                .withMessages(mess)
                                .withName(nameAndConfig.getLeft());
                    } else {
                        return getMessageGroup(mess, nameAndConfig.getRight(), additionalInfo)
                                .withName(nameAndConfig.getLeft())
                                .withMessages(mess);
                    }
                })
                .toList();
    }

    protected CommunicationMessageGroup getMessageGroup(
            List<CommunicationMessage> messages,
            CommunicationEventVersion groupConfig,
            AdditionalInfoContainer additionalInfo
    ) {
        additionalInfo.withParamsSupplier(info -> prepareGroupTextParams(messages, info, groupConfig));
        return new CommunicationMessageGroup().withContent(new CommunicationMessageContent()
                .withImageUrl(groupConfig.getImageHref())
                .withTitle(getParametrizedText(groupConfig.getTitleConfig(), additionalInfo))
                .withText(getParametrizedText(groupConfig.getTextConfig(), additionalInfo))
                .withButtons(getButtons(groupConfig, null, additionalInfo))
                .withDataContainer(getDataContainer(groupConfig.getDataContainer(), additionalInfo))
                .withParams(additionalInfo.getParams().orElse(Collections.emptyMap())));
    }

    protected Map<String, String> prepareGroupTextParams(
            List<CommunicationMessage> messages,
            AdditionalInfoContainer additionalInfo,
            CommunicationEventVersion groupConfig
    ) {
        var multiParams = StreamEx.of(messages)
                .flatMap(m -> m.getContent().getParams().entrySet().stream())
                .mapToEntry(Map.Entry::getKey, Map.Entry::getValue)
                .append(getParamsByAdditionalInfo(additionalInfo))
                .grouping();
        var transformations = nvl(groupConfig.getParamTransformations(), Collections.<String, String>emptyMap());
        return EntryStream.of(multiParams)
                .mapToValue((key, values) -> merge(values, transformations.get(key)))
                .append("message.count", Integer.toString(messages.size()))
                .toMap();
    }

    public Map<SlotMessageId, CommunicationMessage> format(
            List<ObjectEventData> messagesData,
            Map<Long, CommunicationMessageData> caesarDataByMessageId,
            Map<SlotMessageId, CommunicationEventVersion> versionByMessageId,
            Map<Long, Slot> slotById,
            Map<SlotMessageId, Boolean> isActualByMessageId,
            AdditionalInfoContainer additionalInfo
    ) {
        return StreamEx.of(messagesData)
                .mapToEntry(
                        SlotMessageId::of,
                        data -> {
                            var slotMessageId = SlotMessageId.of(data);
                            return format(data,
                                    caesarDataByMessageId.get(slotMessageId.getMessageId()),
                                    versionByMessageId.get(slotMessageId),
                                    slotById.get(slotMessageId.getSlotId()),
                                    isActualByMessageId.get(slotMessageId),
                                    additionalInfo
                            );
                        }
                )
                .nonNullValues()
                .toMap();
    }

    public CommunicationMessage format(
            ObjectEventData objectEventData,
            CommunicationMessageData caesarData,
            CommunicationEventVersion version,
            Slot slot,
            boolean isActual,
            AdditionalInfoContainer additionalInfo
    ) {
        var caesarMessageData = Optional.ofNullable(caesarData)
                .map(CommunicationMessageData::getMessageData);
        var caesarWebMessageData = caesarMessageData
                .map(TMessageData::getDirectWebData);
        var majorVersion = nvl(version.getMajorDataVersion(), 1L);
        var minorVersion = 0L;
        if (caesarWebMessageData.isPresent() && majorVersion == caesarWebMessageData.get().getMajorViewDataVersion()) {
            minorVersion = caesarWebMessageData.get().getMinorViewDataVersion();
            if (!caesarWebMessageData.get().getViewData().equals(
                    convertJsonStringToProto(objectEventData.getData()))) {
                minorVersion++;
            }
        }
        var data = parseDataFromJson(objectEventData.getData(), additionalInfo, version);
        additionalInfo.withParamsSupplier(info -> prepareTextParams(objectEventData, data, info, version));
        return new CommunicationMessage()
                .withRequestId(additionalInfo.getRequestId().orElse(null))
                .withMessageId(objectEventData.getMessageId())
                .withInventoryAdditionalData(objectEventData.getAdditionalData())
                .withClientId(additionalInfo.getClientId().map(ClientId::asLong).orElse(null))
                .withUserId(additionalInfo.getUserId().orElse(null))
                .withSlot(slot)
                .withEventId(version.getEventId())
                .withEventVersionId(version.getIter())
                .withName(version.getName())
                .withGroupName(version.getGroupName())
                .withMajorVersion(majorVersion)
                .withMinorVersion(minorVersion)
                .withSeverity(nvl(version.getSeverity(), Severity.NORMAL))
                .withStatus(getWebUIStatus(isActual, version, slot, caesarMessageData
                        .map(TMessageData::getStatusList).orElse(List.of())))
                .withTargetObject(new CommunicationTargetObject()
                        .withType(entityTypeToObjectType(version.getTargetEntityType())) // взять из конфига CommunicationEvent
                        .withId(objectEventData.getObjectId()))
                .withData(data)
                .withContent(getContent(objectEventData, caesarData, version, slot, isActual, additionalInfo))
                .withMailData(getMailData(objectEventData, caesarData, version, slot, additionalInfo));
    }

    protected YandexSenderTemplateParams getMailData(
            ObjectEventData objectEventData,
            CommunicationMessageData caesarData,
            CommunicationEventVersion version,
            Slot slot,
            AdditionalInfoContainer additionalInfo
    ) {
        var params = additionalInfo.getParams().orElse(Collections.emptyMap());
        var templateId = replaceMacros(version.getMailTemplateId(), params);
        if (templateId == null) {
            return null;
        }
        var args = EntryStream.of(version.getMailTemplateArgs())
                .mapValues(value -> replaceMacros(value, params))
                .nonNullValues()
                .toMap();
        var email = replaceMacros(version.getMailAddressKey(), params);
        return new YandexSenderTemplateParams.Builder()
                .withCampaignSlug(templateId)
                .withToEmail(email)
                .withAsync(Boolean.TRUE)
                .withArgs(args)
                .build();
    }

    protected CommunicationMessageContent getContent(
            ObjectEventData objectEventData,
            CommunicationMessageData caesarData,
            CommunicationEventVersion version,
            Slot slot,
            boolean isActual,
            AdditionalInfoContainer additionalInfo
    ) {
        return new CommunicationMessageContent()
                .withImageUrl(version.getImageHref())
                .withTitle(nullableNvl(version.getTitle(), getParametrizedText(version.getTitleConfig(), additionalInfo)))
                .withText(nullableNvl(version.getText(), getParametrizedText(version.getTextConfig(), additionalInfo)))
                .withButtons(getButtons(version, objectEventData, additionalInfo))
                .withInputFields(getInputFields(version, objectEventData, additionalInfo))
                .withDataContainer(getDataContainer(version.getDataContainer(), additionalInfo))
                .withParams(additionalInfo.getParams().orElse(Collections.emptyMap()));
    }

    protected CommunicationDataContainer getDataContainer(
            DataContainer dataContainer,
            AdditionalInfoContainer additionalInfo
    ) {
        var params = additionalInfo.getParams().orElse(Collections.emptyMap());
        return dataContainer == null ? new CommunicationDataContainer() : new CommunicationDataContainer()
                .withType(dataContainer.getType())
                .withFieldName(dataContainer.getFieldName())
                .withText(nullableNvl(replaceMacros(dataContainer.getText(), params),
                        getParametrizedText(dataContainer.getTextConfig(), additionalInfo)))
                .withSubData(mapList(dataContainer.getSubData(), data -> getDataContainer(data, additionalInfo)));
    }

    protected List<CommunicationMessageInputField> getInputFields(
            CommunicationEventVersion version,
            ObjectEventData objectEventData,
            AdditionalInfoContainer additionalInfo
    ) {
        if (version.getInputFields() == null) {
            return List.of();
        }
        return StreamEx.of(version.getInputFields())
                .map(f -> getInputField(f, version, objectEventData, additionalInfo))
                .toList();
    }

    protected CommunicationMessageInputField getInputField(
            FieldConfig config,
            CommunicationEventVersion version,
            ObjectEventData objectEventData,
            AdditionalInfoContainer additionalInfo
    ) {
        return new CommunicationMessageInputField()
                .withName(config.getName())
                .withTitle(getParametrizedText(config.getTitleConfig(), additionalInfo))
                .withValueType(config.getValueType())
                .withAvailableValues(List.of());
    }

    protected List<CommunicationMessageButton> getButtons(
            CommunicationEventVersion version,
            ObjectEventData objectEventData,
            AdditionalInfoContainer additionalInfo
    ) {
        if (version.getButtonConfigs() == null) {
            return List.of();
        }
        return EntryStream.of(version.getButtonConfigs())
                .map(e -> getButton(
                        version, e.getValue(), objectEventData, additionalInfo
                        ).withId(e.getKey())
                ).toList();
    }

    protected CommunicationMessageButton getButton(
            CommunicationEventVersion version,
            ButtonConfig config,
            ObjectEventData objectEventData,
            AdditionalInfoContainer additionalInfo
    ) {
        var params = additionalInfo.getParams().orElse(Collections.emptyMap());
        return new CommunicationMessageButton()
                .withHref(replaceMacros(config.getHref(), params))
                .withTitle(getParametrizedText(config.getTextConfig(), additionalInfo))
                .withConfirmText(getParametrizedText(config.getConfirmTextConfig(), additionalInfo))
                .withAction(EWebUIButtonActionType.valueOf(config.getAction()))
                .withStyle(EWebUIButtonStyle.valueOf(config.getStyle()))
                .withViewType(nvl(config.getViewType(), ViewType.DEFAULT))
                .withActionType(nvl(config.getActionType(), ActionType.ACTION))
                .withLinkInfo(getLinkInfo(config.getLinkInfo(), additionalInfo))
                .withSuccessTitle(getParametrizedText(config.getSuccessTitleConfig(), additionalInfo))
                .withSuccessText(getParametrizedText(config.getSuccessTextConfig(), additionalInfo))
                .withDisabled(Boolean.FALSE.equals(checkCondition(config.getEnableCondition(), additionalInfo)));
    }

    protected CommunicationLinkInfo getLinkInfo(CommunicationLinkInfoConfig linkInfo, AdditionalInfoContainer additionalInfo) {
        if (linkInfo == null) {
            return null;
        }
        var entityIdsStr = replaceMacros(linkInfo.getEntityIds(),
                additionalInfo.getParams().orElse(Collections.emptyMap()));
        var entityIds = entityIdsStr == null ? null : StreamEx.of(entityIdsStr.split(","))
                .map(Long::parseLong)
                .toList();
        var campaignId = Optional.ofNullable(replaceMacros(linkInfo.getCampaignId(),
                additionalInfo.getParams().orElse(Collections.emptyMap())))
                .map(Long::parseLong)
                .orElse(null);
        return new CommunicationLinkInfo()
                .withPath(nvl(linkInfo.getPath(), LinkPath.GRID))
                .withEntityType(linkInfo.getEntityType())
                .withEntityIds(entityIds)
                .withCampaignId(campaignId)
                .withClientId(additionalInfo.getClientId().orElse(null));
    }

    protected Boolean checkCondition(Condition config, AdditionalInfoContainer additionalInfo) {
        if (config == null) {
            return null;
        } else if (config.getNotSubCondition() != null) {
            return !checkCondition(config.getNotSubCondition(), additionalInfo);
        } else if (config.getAndSubConditions() != null) {
            return StreamEx.of(config.getAndSubConditions())
                    .findAny(cond -> !checkCondition(cond, additionalInfo))
                    .isEmpty();
        } else if (config.getOrSubConditions() != null) {
            return StreamEx.of(config.getOrSubConditions())
                    .findAny(cond -> checkCondition(cond, additionalInfo))
                    .isPresent();
        } else {
            var leftValue = replaceMacros(config.getLeftValue(),
                    additionalInfo.getParams().orElse(Collections.emptyMap()));
            var rightValue = replaceMacros(config.getRightValue(),
                    additionalInfo.getParams().orElse(Collections.emptyMap()));
            return Objects.equals(leftValue, rightValue);
        }
    }

    protected String getParametrizedText(
            TextConfig config,
            AdditionalInfoContainer additionalInfo
    ) {
        String template = getText(config, additionalInfo.getLanguage().orElse("ru"));
        var params = additionalInfo.getParams().orElse(Collections.emptyMap());
        return replaceMacros(template, params);
    }

    protected String replaceMacros(
            String template,
            Map<String, String> params
    ) {
        if (template == null) {
            return null;
        }
        return StringSubstitutor.replace(template, params);
    }

    /**
     * Параметры из расчетов преобразуются в строки, а к ключу добавляется префикс yql.
     * Так же в результат добавляются данные клиента и оператора
     * @param data данные рекомендации из расчетов
     * @param version конфигурация рекомендации
     * @return параметры для подстановки в тексты и для ответа наружу
     */
    protected Map<String, String> prepareTextParams(
            ObjectEventData objectEventData,
            Map<String, Object> data,
            AdditionalInfoContainer additionalInfo,
            CommunicationEventVersion version
    ) {
        return EntryStream.<String, String>empty()
                .append(getParamsByAdditionalInfo(additionalInfo))
                .append(getParamsByObjectEventData(objectEventData, additionalInfo))
                .append(getParamsByDataMap(data, version))
                .toMap();
    }

    protected Map<String, String> getParamsByDataMap(
            Map<String, Object> data,
            CommunicationEventVersion version
    ) {
        var transformations = nvl(version.getParamTransformations(), Collections.<String, String>emptyMap());
        return EntryStream.of(data)
                .mapToValue((k, v) -> transform(v, transformations.get(k)))
                .mapKeys(k -> "yql." + k)
                .nonNullValues()
                .toMap();
    }

    protected Map<String, String> getParamsByObjectEventData(
            ObjectEventData objectEventData,
            AdditionalInfoContainer additionalInfo
    ) {
        var campaign = Optional.ofNullable(additionalInfo
                .getContextCampaigns()
                .orElse(Collections.emptyMap())
                .get(objectEventData == null ? 0L : objectEventData.getObjectId()));
        return EntryStream.<String, String>empty()
                .append("campaign.name", campaign.map(Campaign::getName).orElse(null))
                .append("campaign.id", campaign.map(Campaign::getId).map(Object::toString).orElse(null))
                .nonNullValues()
                .toMap();
    }

    protected Map<String, String> getParamsByAdditionalInfo(AdditionalInfoContainer additionalInfo) {
        var currencyTranslation = additionalInfo.getCurrency()
                .map(Currency::getCode)
                .map(CurrencyCode::getTranslation);
        var locale = additionalInfo.getLanguage().map(Locale::forLanguageTag);
        return EntryStream.<String, String>empty()
                .append("client.id", additionalInfo.getClientId().get().toString())
                .append("client.login", additionalInfo.getClientLogin().orElse(null))
                .append("client.chief.uid", additionalInfo.getClientUid().map(l -> l.toString()).orElse(null))
                .append("client.chief.login", additionalInfo.getClientLogin().orElse(null))
                .append("client.chief.name", additionalInfo.getClientName().orElse(null))
                .append("client.chief.email", additionalInfo.getClientEmail().orElse(null))
                .append("client.currency.code", additionalInfo.getCurrency()
                        .map(Currency::getCode)
                        .map(Enum::toString)
                        .orElse(null))
                .append("client.currency.shortForm", currencyTranslation
                        .map(CurrencyTranslation::shortForm)
                        .map(t -> translate(t, locale))
                        .orElse(null))
                .append("client.currency.symbol", currencyTranslation
                        .map(CurrencyTranslation::symbol)
                        .map(t -> translate(t, locale))
                        .orElse(null))
                .append("operator.login", additionalInfo.getUserLogin().orElse(null))
                .append("operator.canWrite", additionalInfo.getCanUserWrite()
                        .map(b -> b.toString())
                        .orElse(null))
                .nonNullValues()
                .toMap();
    }

    private String translate(Translatable translatable, Optional<Locale> locale) {
        return locale.isEmpty() ? null : translationService.translate(translatable, locale.get());
    }

    public static Map<String, Object> parseDataFromJson(
            String json,
            AdditionalInfoContainer additionalInfo,
            CommunicationEventVersion version) {
        Map<String, JsonNode> map = new HashMap<>();
        Optional.ofNullable(json)
                .map(JsonUtils::fromJson)
                .map(JsonNode::fields)
                .orElse(Map.<String, JsonNode>of().entrySet().iterator())
                .forEachRemaining(e -> map.put(e.getKey(), e.getValue()));
        var currencyData = nvl(
                version.getCurrencyDataFields(),
                List.of("budget_spendings_increase", "new_daily_budget", "new_week_budget"));
        return EntryStream.of(map)
                .mapToValue((key, node) -> {
                    if (node == null || node.isNull()) {
                        return (Object) null;
                    } else if (currencyData.contains(key)) {
                        var nds = additionalInfo.getNds()
                                .orElse(Percent.fromPercent(BigDecimal.ZERO));
                        var ratio = additionalInfo.getCurrency().map(Currency::getYabsRatio).orElse(1);
                        return BigDecimal.valueOf(
                                node.asLong() / 1000000.0 * ratio).divide(nds.asRatio().add(BigDecimal.ONE), 2, RoundingMode.HALF_UP);
                    } else if (node.isBigDecimal() || node.isDouble() || node.isFloat()) {
                        return BigDecimal.valueOf(node.asDouble());
                    } else if (node.isNumber()) {
                        return node.asLong();
                    } else if (node.isArray() || node.isContainerNode()) {
                        return node;
                    } else {
                        return node.asText();
                    }
                })
                .append("client.currency", additionalInfo.getCurrency().orElse(null))
                .nonNullValues()
                .toMap();
    }

    protected String getText(
            TextConfig config,
            String lang
    ) {
        if (config == null) {
            return null;
        }
        if (config.getTextByLocale() != null) {
            String result = config.getTextByLocale().get(lang);
            return result != null ? result : config.getTextByLocale().get("en");
        }
        return translationService.translate(
                config.getTranslateBundleName(),
                config.getTranslateKey(),
                Locale.forLanguageTag(lang)
        );
    }

    private static EWebUIMessageStatus getWebUIStatus(
            boolean isActual, CommunicationEventVersion version,
            Slot slot, List<EMessageStatus> statuses)
    {
        if (statuses.contains(EMessageStatus.REJECT)) {
            return EWebUIMessageStatus.MESSAGE_STATUS_REJECTED;
        } else if (!isActual) {
            return EWebUIMessageStatus.MESSAGE_STATUS_DONE;
        } else if (slot.getShowAll() && isEventOptional(version)) {
            return EWebUIMessageStatus.MESSAGE_STATUS_OPTIONAL;
        } else if (statuses.isEmpty()) {
            return EWebUIMessageStatus.MESSAGE_STATUS_NEW;
        } else {
            return EWebUIMessageStatus.MESSAGE_STATUS_ACTUAL;
        }
    }

    @Override
    public List<CommunicationMessage> format(AutoUpdatedSettingsEvent event, CommunicationEventVersion version,
                                       List<Slot> slots, AdditionalInfoContainer additionalInfo) {
        var operatorUid = additionalInfo.getUserId().orElse(null);
        var clientId = additionalInfo.getClientId().map(ClientId::asLong).orElseThrow();
        var data = new HashMap<String, Object>();
        data.put("history.new_value", event.getNewValue());
        data.put("history.old_value", event.getOldValue());
        additionalInfo.withParamsSupplier(info -> prepareTextParams(null, data, info, version));
        return mapList(slots, slot -> new CommunicationMessage()
                .withUserId(operatorUid)
                .withClientId(clientId)
                .withEventId(version.getEventId())
                .withEventVersionId(version.getIter())
                .withName(version.getName())
                .withGroupName(version.getGroupName())
                .withMajorVersion(version.getMajorDataVersion())
                .withMinorVersion(0)
                .withDisableForEdit(event.isAutoApplyEnabled()) // включена ли сейчас опция на кампании
                .withStatus(EWebUIMessageStatus.MESSAGE_STATUS_ACTUAL)
                .withSlot(slot)
                .withMessageId(CommunicationHelper.calculateMessageId(operatorUid,
                        additionalInfo.getClientId().orElseThrow(),
                        event.getTargetObjectId(),
                        version.getEventId()))
                .withTargetObject(new CommunicationTargetObject()
                        .withId(event.getTargetObjectId())
                        .withType(ETargetObjectType.TARGET_OBJECT_COMPANY))
                .withContent(new CommunicationMessageContent()
                        .withhApplyDateTime(event.getLastAutoUpdateTime())
                        .withTitle(getParametrizedText(version.getTitleConfig(), additionalInfo))
                        .withText(getParametrizedText(version.getTextConfig(), additionalInfo))
                        .withDataContainer(getDataContainer(version.getDataContainer(), additionalInfo))
                        .withButtons(List.of())));
    }

    private static boolean isEventOptional(CommunicationEventVersion version) {
        return Boolean.TRUE.equals(version.getOptional());
    }

    @Override
    public String getName() {
        return DEFAULT_IMPLEMENTATION;
    }
}
