package ru.yandex.direct.communication.facade.impl.processors;

import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;

import com.google.protobuf.MessageLite;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.ads.bsyeti.libs.communications.EMessageStatus;
import ru.yandex.ads.bsyeti.libs.communications.TEventSource;
import ru.yandex.ads.bsyeti.libs.events.TCommunicationEvent;
import ru.yandex.ads.bsyeti.libs.events.TCommunicationEventData;
import ru.yandex.direct.communication.CommunicationClient;
import ru.yandex.direct.communication.container.AdditionalInfoContainer;
import ru.yandex.direct.communication.facade.CommunicationEventVersionsProcessor;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersion;
import ru.yandex.direct.dbutil.model.ClientId;

import static ru.yandex.ads.bsyeti.libs.communications.ECommunicationType.INFORMATION;
import static ru.yandex.ads.bsyeti.libs.communications.EEventType.CREATE_OR_UPDATE;
import static ru.yandex.ads.bsyeti.libs.communications.ESourceType.DIRECT_OFFLINE_REGULAR;
import static ru.yandex.direct.communication.facade.CommunicationEventVersionProcessingFacade.ON_CLIENT_CREATE_TRIGGER;

@Component
public class NewEventProducer implements CommunicationEventVersionsProcessor {

    private static final Logger logger = LoggerFactory.getLogger(NewEventProducer.class);

    private static final long DEFAULT_EVENT_TTL = 60 * 60 * 24 * 30; // месяц
    private static final List<Long> SLOT_TYPES = List.of(1L);

    private final CommunicationClient communicationClient;

    @Autowired
    public NewEventProducer(CommunicationClient communicationClient) {
        this.communicationClient = communicationClient;
    }

    @Override
    public String getName() {
        return "ProduceNewEvent";
    }

    @Override
    public void process(AdditionalInfoContainer target, List<CommunicationEventVersion> eventVersions,
                        String trigger) {
        if (!ON_CLIENT_CREATE_TRIGGER.equals(trigger)) {
            logger.warn(String.format("Skip unsupported trigger %s.", trigger));
            return;
        }
        List<MessageLite> events = StreamEx.of(eventVersions)
                .map(v -> createEvent(target, v))
                .toList();
        try {
            communicationClient.send(events);
        } catch (ExecutionException ex) {
            logger.error("Caught exception", ex);
        } catch (TimeoutException ex) {
            logger.error("Timeout", ex);
        }
    }

    private MessageLite createEvent(AdditionalInfoContainer target, CommunicationEventVersion version) {
        long currentTimeStamp = System.currentTimeMillis() / 1000;
        var clientId = target.getClientId().map(ClientId::asLong).orElse(null);
        var builder = TCommunicationEvent.newBuilder()
                .addUids(clientId)
                .setTimestamp(currentTimeStamp)
                .setData(TCommunicationEventData.newBuilder()
                        .setType(CREATE_OR_UPDATE)
                        .setCommunicationType(INFORMATION)
                        .setId(version.getEventId().intValue())
                        .setTargetEntityId(clientId)
                        .setSource(TEventSource.newBuilder()
                                .setType(DIRECT_OFFLINE_REGULAR)
                                .setId(version.getIter()))
                        .setCreateOrUpdate(TCommunicationEventData.TCreateOrUpdate.newBuilder()
                                .setExpired(currentTimeStamp + DEFAULT_EVENT_TTL)
                                .setNextShowTime(currentTimeStamp)
                                .addAllSlotTypes(SLOT_TYPES)
                                .setMajorViewDataVersion(1L)
                                .setDefaultAction(TCommunicationEventData.TAction.newBuilder() // По умолчанию ставим статус NEW
                                        .addAddStatuses(EMessageStatus.NEW))
                                .setOnMajorVersionChange(TCommunicationEventData.TAction.newBuilder() // По умолчанию убираем все меняющиеся статусы
                                        .addRemoveStatuses(EMessageStatus.MUTE)
                                        .addRemoveStatuses(EMessageStatus.DELIVERED)
                                        .addRemoveStatuses(EMessageStatus.APPLY)
                                        .addRemoveStatuses(EMessageStatus.REJECT))));
        return builder.build();
    }
}
