package ru.yandex.direct.communication.facade.impl.sender;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;

import com.google.protobuf.MessageLite;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.ads.bsyeti.libs.communications.TEventSource;
import ru.yandex.ads.bsyeti.libs.events.TCommunicationEvent;
import ru.yandex.ads.bsyeti.libs.events.TCommunicationEventData;
import ru.yandex.direct.communication.CommunicationClient;
import ru.yandex.direct.communication.container.AdditionalInfoContainer;
import ru.yandex.direct.communication.container.web.CommunicationMessage;
import ru.yandex.direct.communication.facade.MessageSender;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersion;
import ru.yandex.direct.sender.YandexSenderClient;

import static ru.yandex.ads.bsyeti.libs.communications.ECommunicationType.INFORMATION;
import static ru.yandex.ads.bsyeti.libs.communications.EEventType.CONDITIONAL_UPDATE;
import static ru.yandex.ads.bsyeti.libs.communications.ESourceType.RECCOMENDATION_RUNTIME_SERVICE;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
public class MailMessageSender implements MessageSender {

    private static final Logger logger = LoggerFactory.getLogger(MailMessageSender.class);

    private final YandexSenderClient senderClient;
    private final CommunicationClient communicationClient;

    @Autowired
    public MailMessageSender(
            YandexSenderClient senderClient,
            CommunicationClient communicationClient
    ) {
        this.senderClient = senderClient;
        this.communicationClient = communicationClient;
    }

    @Override
    public void send(
            List<CommunicationMessage> messages,
            AdditionalInfoContainer additionalInfo,
            Map<Long, CommunicationEventVersion> versionByMessageId
    ) {
        var sendMessages = new ArrayList<CommunicationMessage>(messages.size());
        messages.forEach(message -> {
            var params = message.getMailData();
            logger.info("Send template " + params.getCampaignSlug() +
                    " to email " + params.getToEmail() + " with args " + params.getArgs());
            if (senderClient.sendTemplate(message.getMailData(), YandexSenderClient::isInvalidToEmail)) {
                sendMessages.add(message);
            } else {
                logger.error("Sending failed");
            }
        });
        List<MessageLite> eventsToSend = mapList(sendMessages, m -> TCommunicationEvent.newBuilder()
                .addUids(m.getClientId())
                .setTimestamp(additionalInfo.getCurrentTimeStamp().get())
                .setData(TCommunicationEventData.newBuilder()
                        .setType(CONDITIONAL_UPDATE)
                        .setCommunicationType(INFORMATION)
                        .setId((int) m.getEventId())
                        .setTargetEntityId(m.getTargetObject().getId())
                        .setSource(TEventSource.newBuilder()
                                .setType(RECCOMENDATION_RUNTIME_SERVICE)
                                .setId(m.getEventVersionId()))
                        .setConditionalUpdate(TCommunicationEventData.TConditionalUpdate.newBuilder().addConditionAndUpdate(
                                TCommunicationEventData.TConditionalUpdate.TConditionAndUpdate.newBuilder()
                                        .setConditionMajorVersion(m.getMajorVersion())
                                        .setConditionMinorVersion(m.getMinorVersion())
                                        .setAction(TCommunicationEventData.TAction.newBuilder()
                                                .setUpdateNextShowTime(additionalInfo.getCurrentTimeStamp().get() +
                                                        versionByMessageId.get(m.getMessageId()).getTimeToLive())))))
                .build());
        try{
            communicationClient.send(eventsToSend);
        } catch (ExecutionException | TimeoutException ex) {
            throw new RuntimeException(ex);
        }
    }

    @Override
    public String getName() {
        return "MAIL";
    }
}
