package ru.yandex.direct.communication.facade.impl.status;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import com.google.protobuf.MessageLite;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.ads.bsyeti.libs.communications.EMessageStatus;
import ru.yandex.ads.bsyeti.libs.communications.TDirectWebUIData;
import ru.yandex.ads.bsyeti.libs.communications.TEventSource;
import ru.yandex.ads.bsyeti.libs.communications.TKeyValues;
import ru.yandex.ads.bsyeti.libs.communications.proto.TMessageData;
import ru.yandex.ads.bsyeti.libs.events.TCommunicationEvent;
import ru.yandex.ads.bsyeti.libs.events.TCommunicationEventData;
import ru.yandex.direct.communication.CommunicationClient;
import ru.yandex.direct.communication.container.AdditionalInfoContainer;
import ru.yandex.direct.communication.container.CommunicationMessageData;
import ru.yandex.direct.communication.container.web.SlotMessageId;
import ru.yandex.direct.communication.facade.StatusChecker;
import ru.yandex.direct.communication.facade.impl.logging.RecommendationLogger;
import ru.yandex.direct.communication.model.Slot;
import ru.yandex.direct.communication.model.inventory.ObjectEventData;
import ru.yandex.direct.core.entity.communication.model.ChangeAction;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersion;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersionStatus;
import ru.yandex.direct.core.entity.communication.model.TargetEntityType;
import ru.yandex.direct.dbutil.model.ClientId;

import static ru.yandex.ads.bsyeti.libs.communications.ECommunicationType.INFORMATION;
import static ru.yandex.ads.bsyeti.libs.communications.EEventType.CREATE_OR_UPDATE;
import static ru.yandex.ads.bsyeti.libs.communications.ESourceType.RECCOMENDATION_RUNTIME_SERVICE;
import static ru.yandex.direct.communication.CommunicationHelper.DEFAULT_ON_ACTUAL;
import static ru.yandex.direct.communication.CommunicationHelper.DEFAULT_ON_MAJOR_VERSION_CHANGE;
import static ru.yandex.direct.communication.CommunicationHelper.DEFAULT_ON_MINOR_VERSION_CHANGE;
import static ru.yandex.direct.communication.CommunicationHelper.DEFAULT_ON_NOT_ACTUAL;
import static ru.yandex.direct.communication.CommunicationHelper.DEFAULT_RESTRICTION;
import static ru.yandex.direct.communication.CommunicationHelper.convertJsonStringToProto;
import static ru.yandex.direct.communication.facade.CommunicationEventVersionProcessingFacade.DEFAULT_IMPLEMENTATION;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
public class DefaultStatusChecker implements StatusChecker {

    private static final Logger logger = LoggerFactory.getLogger(DefaultStatusChecker.class);

    private final CommunicationClient communicationClient;

    @Autowired
    public DefaultStatusChecker(CommunicationClient communicationClient) {
        this.communicationClient = communicationClient;
    }

    @Override
    public String getName() {
        return DEFAULT_IMPLEMENTATION;
    }

    @Override
    public Set<SlotMessageId> filterHidden(
            List<ObjectEventData> messagesData,
            Map<Long, CommunicationMessageData> caesarDataByMessageId,
            Map<SlotMessageId, CommunicationEventVersion> versionByMessageId,
            Map<Long, Slot> slotById,
            Map<SlotMessageId, Boolean> isActualByMessageId,
            AdditionalInfoContainer additionalInfo
    ) {
        List<MessageLite> eventToSend = new ArrayList<>();
        var result =  StreamEx.of(messagesData)
                .filter(data -> {
                    var mId = SlotMessageId.of(data);
                    return checkVisibilityAndCollectEventsToSend(eventToSend, data,
                            Optional.ofNullable(caesarDataByMessageId.get(mId.getMessageId())),
                            versionByMessageId.get(mId),
                            slotById.get(mId.getSlotId()),
                            isActualByMessageId.get(mId),
                            additionalInfo
                    );
                })
                .toMap(
                        SlotMessageId::of,
                        Function.identity()
                );
        result.values().forEach(data -> RecommendationLogger.logSelect(data, slotById.get(data.getSlotId()),
                additionalInfo));

        try {
            communicationClient.send(eventToSend);
        } catch (Exception ex) {
            logger.error("Caught exception", ex);
        }
        return result.keySet();
    }

    /**
     * Првоеряет необходимость показа сообщения и добавляет в очередь отправки сообщения для Цезаря, если это необходимо
     * @param eventsToSend очередь событий в Цезарь
     * @param objectEventData данные сообщения
     * @param caesarData данные по событию из Цезаря
     * @param version конфигурация версии события
     * @param slot конфигурация слота
     * @param isActual актуально ли сообщение
     * @param additionalInfo дополнительные данные запроса
     * @return true если сообщение нужно отдать пользователю
     */
    public boolean checkVisibilityAndCollectEventsToSend(
            Collection eventsToSend,
            ObjectEventData objectEventData,
            Optional<CommunicationMessageData> caesarData,
            CommunicationEventVersion version,
            Slot slot,
            boolean isActual,
            AdditionalInfoContainer additionalInfo
    ) {
        if (caesarData.isEmpty() || isActual) {
            eventsToSend.add(buildCreateOrUpdateEvent(objectEventData, version, additionalInfo));
            RecommendationLogger.logCreateOrUpdate(objectEventData, slot, additionalInfo, caesarData.isEmpty());
        }
        if (!isActual) {
            eventsToSend.add(buildMuteEvent(objectEventData, version, additionalInfo));
            RecommendationLogger.logMute(objectEventData, slot, additionalInfo);
        }
        if (!isActual && !slot.getShowAll()) {
            RecommendationLogger.logSkip(objectEventData, slot, additionalInfo, "NOT_ACTUAL");
            return false;
        }
        if (version.getStatus() != CommunicationEventVersionStatus.ACTIVE ||
                version.getFeatures() != null &&
                        !additionalInfo.getEnabledFeatures().orElse(Collections.emptyList())
                                .containsAll(version.getFeatures())) {
            RecommendationLogger.logSkip(objectEventData, slot, additionalInfo, "DISABLED");
            return false;
        }

        var caesarMessageData = caesarData
                .map(CommunicationMessageData::getMessageData);
        var caesarWebMessageData = caesarMessageData
                .map(TMessageData::getDirectWebData);

        //Проверяем, какие изменения будут в Цезаре из-за изменения данных рекомендации
        ChangeAction action;
        if (!caesarWebMessageData.map(TDirectWebUIData::getMajorViewDataVersion)
                .orElse(0L).equals(nvl(version.getMajorDataVersion(), 1L))) {
            action = nvl(version.getOnMajorVersionChange(), DEFAULT_ON_MAJOR_VERSION_CHANGE);
        } else if (!caesarWebMessageData.map(TDirectWebUIData::getViewData)
                .orElse(TKeyValues.getDefaultInstance())
                .equals(convertJsonStringToProto(objectEventData.getData()))) {
            action = nvl(version.getOnMinorVersionChange(), DEFAULT_ON_MINOR_VERSION_CHANGE);
        } else {
            action = nvl(version.getOnActual(), DEFAULT_ON_ACTUAL);
        }

        final long now = additionalInfo.getCurrentTimeStamp().orElseThrow();
        //Берем из Цезаря время следующего показа и статусы
        long nextShowTime = caesarData
                .map(CommunicationMessageData::getNextShowTime)
                .orElse(now);
        var statuses = new HashSet<>(caesarMessageData.map(TMessageData::getStatusList)
                .orElse(Collections.emptyList()));

        // Учитываем изменение статусов и времени следующего показа из-за изменения данных рекомендации
        if (action.getUpdateNextShowTime() != null) {
            nextShowTime = now + action.getUpdateNextShowTime();
        }
        if (action.getRemoveStatuses() != null) {
            statuses.removeAll(mapList(action.getRemoveStatuses(), EMessageStatus::valueOf));
        }
        if (action.getAddStatuses() != null) {
            statuses.addAll(mapList(action.getAddStatuses(), EMessageStatus::valueOf));
        }
        // Учитываем изменения статусов, которые сохранены в куке, но еще не доехали до Цезаря
        statuses.addAll(additionalInfo.getCookieAddedStatusesByMessageId().orElse(Collections.emptyMap())
                .getOrDefault(objectEventData.getMessageId(), Collections.emptySet()));
        statuses.removeAll(additionalInfo.getCookieRemovedStatusesByMessageId().orElse(Collections.emptyMap())
                .getOrDefault(objectEventData.getMessageId(), Collections.emptySet()));

        // Проверяем выставленный таймаут на показ
        if (nextShowTime > now) {
            RecommendationLogger.logSkip(objectEventData, slot, additionalInfo, "TIMEOUT");
            return false;
        }

        // Проверяем статусы, останавливаемые показ
        var restrictions = DEFAULT_RESTRICTION;
        if (version.getRestrinctionBySlot() != null) {
            for (var slotId : List.of(slot.getId(), 0L)) {
                if (version.getRestrinctionBySlot().containsKey(slotId)) {
                    restrictions = version.getRestrinctionBySlot().get(slotId);
                    break;
                }
            }
        }
        var stopStatus = getStopStatus(statuses, restrictions.getStopStatuses());
        if (stopStatus.isPresent()) {
            RecommendationLogger.logSkip(objectEventData, slot, additionalInfo, stopStatus.get());
            return false;
        }

        return true;
    }

    private static Optional<EMessageStatus> getStopStatus(
            Collection<EMessageStatus> statuses,
            List<String> stoppedStatuses
    ) {
        return StreamEx.of(stoppedStatuses)
                .map(EMessageStatus::valueOf)
                .findFirst(statuses::contains);
    }

    private MessageLite buildMuteEvent(
            ObjectEventData objectEventData,
            CommunicationEventVersion version,
            AdditionalInfoContainer additionalInfo
    ) {
        var clientIdOrUserId = TargetEntityType.USER.equals(version.getTargetEntityType()) ?
                additionalInfo.getUserId().orElse(null) :
                additionalInfo.getClientId().map(ClientId::asLong).orElse(null);
        return TCommunicationEvent.newBuilder()
                .addUids(clientIdOrUserId)
                .setTimestamp(additionalInfo.getCurrentTimeStamp().orElseThrow())
                .setData(TCommunicationEventData.newBuilder()
                        .setType(CREATE_OR_UPDATE)
                        .setCommunicationType(INFORMATION)
                        .setId((int) objectEventData.getEventId())
                        .setTargetEntityId(objectEventData.getObjectId())
                        .setSource(TEventSource.newBuilder()
                                .setType(RECCOMENDATION_RUNTIME_SERVICE)
                                .setId(objectEventData.getEventVersionId()))
                        .setConditionalUpdate(TCommunicationEventData.TConditionalUpdate.newBuilder()
                                .addConditionAndUpdate(TCommunicationEventData.TConditionalUpdate.TConditionAndUpdate
                                        .newBuilder().setAction(getAction(
                                                nvl(version.getOnNotActual(), DEFAULT_ON_NOT_ACTUAL),
                                                additionalInfo)))))
                .build();
    }

    private MessageLite buildCreateOrUpdateEvent(
            ObjectEventData objectEventData,
            CommunicationEventVersion version,
            AdditionalInfoContainer additionalInfo
    ) {
        var clientIdOrUserId = TargetEntityType.USER.equals(version.getTargetEntityType()) ?
                additionalInfo.getUserId().orElse(null) :
                additionalInfo.getClientId().map(ClientId::asLong).orElse(null);
        long now = additionalInfo.getCurrentTimeStamp().orElseThrow();
        return TCommunicationEvent.newBuilder()
                .addUids(clientIdOrUserId)
                .setTimestamp(now)
                .setData(TCommunicationEventData.newBuilder()
                        .setType(CREATE_OR_UPDATE)
                        .setCommunicationType(INFORMATION)
                        .setId((int) objectEventData.getEventId())
                        .setTargetEntityId(objectEventData.getObjectId())
                        .setSource(TEventSource.newBuilder()
                                .setType(RECCOMENDATION_RUNTIME_SERVICE)
                                .setId(objectEventData.getEventVersionId()))
                        .setCreateOrUpdate(TCommunicationEventData.TCreateOrUpdate.newBuilder()
                                .setExpired(now + nvl(version.getTimeToLive(), 0L))
                                .setNextShowTime(now)
                                .setViewData(convertJsonStringToProto(objectEventData.getData()))
                                .addAllSlotTypes(version.getSlots())
                                .setMajorViewDataVersion(nvl(version.getMajorDataVersion(), 1L))
                                .setDefaultAction(getAction(
                                        nvl(version.getOnActual(), DEFAULT_ON_ACTUAL),
                                        additionalInfo))
                                .setOnMinorVersionChange(getAction(
                                        nvl(version.getOnMinorVersionChange(), DEFAULT_ON_MINOR_VERSION_CHANGE),
                                        additionalInfo))
                                .setOnMajorVersionChange(getAction(
                                        nvl(version.getOnMajorVersionChange(), DEFAULT_ON_MAJOR_VERSION_CHANGE),
                                        additionalInfo))
                        ))
                .build();
    }

    private static TCommunicationEventData.TAction.Builder getAction(
            ChangeAction actionConfig,
            AdditionalInfoContainer additionalInfo
    ) {
        var builder = TCommunicationEventData.TAction.newBuilder();
        if (actionConfig.getUpdateNextShowTime() != null) {
            builder.setUpdateNextShowTime(additionalInfo.getCurrentTimeStamp().orElseThrow() +
                    actionConfig.getUpdateNextShowTime());
        }
        if (actionConfig.getAddStatuses() != null) {
            actionConfig.getAddStatuses().forEach(status -> builder
                    .addAddStatuses(EMessageStatus.valueOf(status)));
        }
        if (actionConfig.getRemoveStatuses() != null) {
            actionConfig.getRemoveStatuses().forEach(status -> builder
                    .addRemoveStatuses(EMessageStatus.valueOf(status)));
        }
        return builder;
    }
}
