package ru.yandex.direct.communication.service;

import java.math.BigDecimal;
import java.time.Duration;
import java.time.LocalDate;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.ads.bsyeti.libs.communications.EMessageStatus;
import ru.yandex.ads.bsyeti.libs.communications.EWebUIButtonActionType;
import ru.yandex.ads.bsyeti.libs.communications.EWebUIButtonStyle;
import ru.yandex.direct.autobudget.restart.model.CampStrategyRestartResult;
import ru.yandex.direct.autobudget.restart.model.CampStrategyRestartResultSuccess;
import ru.yandex.direct.autobudget.restart.model.StrategyDto;
import ru.yandex.direct.autobudget.restart.service.AutobudgetRestartService;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.communication.CommunicationChannelRepository;
import ru.yandex.direct.communication.CommunicationHelper;
import ru.yandex.direct.communication.container.AdditionalInfoContainer;
import ru.yandex.direct.communication.container.CommunicationMessageData;
import ru.yandex.direct.communication.container.web.CommunicationMessage;
import ru.yandex.direct.communication.container.web.CommunicationMessageGroup;
import ru.yandex.direct.communication.container.web.SlotMessageId;
import ru.yandex.direct.communication.facade.ActionResult;
import ru.yandex.direct.communication.facade.ActionTarget;
import ru.yandex.direct.communication.facade.CommunicationEventVersionProcessingFacade;
import ru.yandex.direct.communication.facade.SummaryActionResult;
import ru.yandex.direct.communication.facade.impl.logging.RecommendationLogger;
import ru.yandex.direct.communication.inventory.CommunicationInventoryClient;
import ru.yandex.direct.communication.model.Slot;
import ru.yandex.direct.communication.model.inventory.ETriggerType;
import ru.yandex.direct.communication.model.inventory.ObjectDataRequest;
import ru.yandex.direct.communication.model.inventory.ObjectEventData;
import ru.yandex.direct.communication.model.inventory.Request;
import ru.yandex.direct.communication.model.inventory.Response;
import ru.yandex.direct.communication.model.inventory.SlotRequest;
import ru.yandex.direct.communication.model.inventory.SlotResponse;
import ru.yandex.direct.communication.repository.CommunicationSlotRepository;
import ru.yandex.direct.core.entity.campaign.container.CampaignsSelectionCriteria;
import ru.yandex.direct.core.entity.campaign.model.CampOptionsStrategy;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignOpts;
import ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.validation.type.disabled.DisabledField;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.model.ClientNds;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.core.entity.client.service.ClientNdsService;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersion;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersionStatus;
import ru.yandex.direct.core.entity.communication.repository.CommunicationEventVersionsRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.metrika.repository.MetrikaCampaignRepository;
import ru.yandex.direct.core.entity.recommendation.model.AutoChangeableSettings;
import ru.yandex.direct.core.entity.recommendation.model.RecommendationQueueInfo;
import ru.yandex.direct.core.entity.user.repository.UserRepository;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Percent;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.libs.timetarget.TimeTarget;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.useractionlog.model.AutoUpdatedSettingsEvent;
import ru.yandex.direct.useractionlog.reader.LastAutoUpdatedSettingsActionLogReader;
import ru.yandex.direct.utils.CollectionUtils;
import ru.yandex.direct.utils.FunctionalUtils;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds.WEB_EDIT;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class CommunicationChannelService {

    private static final Logger logger = LoggerFactory.getLogger(CommunicationChannelService.class);

    private static final Set<String> RECOMMENDATION_FEATURE_NAMES = FunctionalUtils.mapSet(Set.of(
            FeatureName.RECOMMEND_CLICK_DAILY_BUDGET,
            FeatureName.RECOMMEND_CASHBACK_LAST_MONTH,
            FeatureName.RECOMMEND_UC_NEW_CPM,
            FeatureName.RECOMMEND_UC_CAMP_GOALS,
            FeatureName.ENABLE_NEW_DAILY_BUDGET_RECOMMENDATION,
            FeatureName.ENABLE_NEW_WEEKLY_BUDGET_RECOMMENDATION,
            FeatureName.ENABLE_AUTO_APPLY_RECOMMENDATION
    ), FeatureName::getName);
    private static final LocalDate OLD_START_TIME = LocalDate.parse("2000-01-01");

    public static final String NOT_ACTUAL = "Not actual";
    public static final String WAS_CHANGED = "was changed";
    public static final String NOT_SUPPORTED = "Not supported";

    public static final String RU = "ru";

    private final ShardHelper shardHelper;
    private final PpcProperty<Boolean> ignoreAutobudgetRestartProp;
    private final FeatureService featureService;
    private final ClientRepository clientRepository;
    private final ClientNdsService clientNdsService;
    private final UserRepository userRepository;
    private final RbacService rbacService;
    private final CommunicationChannelRepository channelRepository;
    private final CommunicationEventVersionsRepository eventVersionsRepository;
    private final CommunicationSlotRepository slotRepository;
    private final CommunicationEventVersionProcessingFacade processingFacade;
    private final CommunicationInventoryClient inventoryClient;
    private final AutobudgetRestartService autobudgetRestartService;
    private final CampaignRepository campaignRepository;
    private final MetrikaCampaignRepository metrikaCampaignRepository;
    private final LastAutoUpdatedSettingsActionLogReader userActionLogReader;
    private final PpcProperty<Integer> maxTimeDepthForAutoApplyHistory;

    @Autowired
    public CommunicationChannelService(
            ShardHelper shardHelper,
            PpcPropertiesSupport propertiesSupport,
            FeatureService featureService,
            ClientRepository clientRepository,
            ClientNdsService clientNdsService,
            UserRepository userRepository,
            RbacService rbacService,
            CommunicationChannelRepository communicationChannelRepository,
            CommunicationEventVersionsRepository communicationEventVersionsRepository,
            CommunicationSlotRepository communicationSlotRepository,
            CommunicationEventVersionProcessingFacade communicationEventVersionProcessingFacade,
            CommunicationInventoryClient communicationInventoryClient,
            AutobudgetRestartService autobudgetRestartService,
            CampaignRepository campaignRepository,
            MetrikaCampaignRepository metrikaCampaignRepository,
            LastAutoUpdatedSettingsActionLogReader userActionLogReader
    ) {
        this.shardHelper = shardHelper;
        ignoreAutobudgetRestartProp = propertiesSupport.get(PpcPropertyNames.IGNORE_AUTOBUDGET_RESTART,
                Duration.ofMinutes(5));
        maxTimeDepthForAutoApplyHistory = propertiesSupport.get(PpcPropertyNames.MAX_TIME_DEPTH_FOR_AUTO_APPLY_HISTORY,
                Duration.ofMinutes(5));
        this.featureService = featureService;
        this.clientRepository = clientRepository;
        this.clientNdsService = clientNdsService;
        this.userRepository = userRepository;
        this.rbacService = rbacService;
        channelRepository = communicationChannelRepository;
        eventVersionsRepository = communicationEventVersionsRepository;
        slotRepository = communicationSlotRepository;
        processingFacade = communicationEventVersionProcessingFacade;
        inventoryClient = communicationInventoryClient;
        this.autobudgetRestartService = autobudgetRestartService;
        this.campaignRepository = campaignRepository;
        this.metrikaCampaignRepository = metrikaCampaignRepository;
        this.userActionLogReader = userActionLogReader;
    }

    private Response getCalculatedMessages(
            AdditionalInfoContainer additionalInfo,
            Set<Long> slots,
            List<Long> targetObjectIds,
            ETriggerType triggerType,
            List<CampStrategyRestartResultSuccess> autoBudgetRestartTimes
    ) {
        if (slots.isEmpty()) {
            return Response.newBuilder().build();
        }
        var restartTimes = listToMap(
                autoBudgetRestartTimes,
                CampStrategyRestartResult::getCid);

        var objectDataRequests = mapList(
                targetObjectIds,
                objectId -> {
                    var builder = ObjectDataRequest.newBuilder()
                            .setId(objectId);
                    var restartTime = restartTimes.get(objectId);
                    if (restartTime != null) {
                        builder.setAutobudgetStartTime(restartTime
                                .getRestartTime().atZone(ZoneId.systemDefault()).toEpochSecond());
                        builder.setAutobudgetLastUpdateTime(restartTime
                                .getSoftRestartTime().atZone(ZoneId.systemDefault()).toEpochSecond());
                    }
                    return builder.build();
                });
        var slotRequests = mapList(
                slots,
                slotId -> SlotRequest.newBuilder()
                        .setSlotId(slotId)
                        .addAllObjects(objectDataRequests)
                        .build());
        Request request = getRequest(additionalInfo, triggerType, slotRequests);
        var result = inventoryClient.getRecommendations(request);
        var builder = result.toBuilder();
        builder.getSlotResponsesBuilderList()
                .forEach(slotResponse -> slotResponse
                        .getObjectEventDataBuilderList().forEach(objectEventData -> objectEventData
                                .setMessageId(CommunicationHelper
                                        .calculateMessageId(
                                                additionalInfo.getUserId().orElse(null),
                                                additionalInfo.getClientId().orElse(null),
                                                objectEventData.getObjectId(),
                                                objectEventData.getEventId()))
                                .setSlotId(slotResponse.getSlotId())));
        logger.info("Communication response with messageIds: " + builder.build());
        return builder.build();
    }

    @NotNull
    private Request getRequest(AdditionalInfoContainer additionalInfo, ETriggerType triggerType,
                               List<SlotRequest> slotRequests) {
        var requestBuilder = Request.newBuilder()
                .setEventRequestTriggerType(triggerType)
                .addAllSlotRequests(slotRequests);
        additionalInfo.getRequestId().ifPresent(requestBuilder::setRequestId);
        additionalInfo.getClientId().map(ClientId::asLong).ifPresent(requestBuilder::setClientId);
        additionalInfo.getUserId().ifPresent(requestBuilder::setOperatorUid);
        additionalInfo.getClientLogin().ifPresent(requestBuilder::setClientLogin);
        additionalInfo.getUserLogin().ifPresent(requestBuilder::setOperatorLogin);
        additionalInfo.getEnabledFeatures().ifPresent(requestBuilder::addAllEnabledFeatures);
        additionalInfo.getTestIds().ifPresent(requestBuilder::addAllTestIds);
        return requestBuilder.build();
    }

    public boolean apply(RecommendationQueueInfo recommendation){
        var eventVersion = recommendation.getUserKey1().split("-", 2);
        var dataVersion = recommendation.getUserKey2().split("-", 2);
        var reqSlotKey = recommendation.getUserKey3().split("-", 2);

        return apply(
                Long.parseLong(reqSlotKey[0]),
                ClientId.fromLong(recommendation.getClientId()),
                recommendation.getUid(),
                recommendation.getCampaignId(),
                reqSlotKey[1],
                Long.parseLong(eventVersion[0]),
                Long.parseLong(eventVersion[1]),
                Long.parseLong(dataVersion[0]),
                Long.parseLong(dataVersion[1])
        );
    }

    public boolean apply(
            Long requestId,
            ClientId clientId,
            Long operatorUid,
            Long targetObjectId,
            String slotName,
            Long eventId,
            Long eventVersionId,
            Long majorDataVersion,
            Long minorDataVersion
    ) {
        var slot = slotRepository.getSlotsByNames(List.of(slotName)).get(0);
        var additionalInfo = getAdditionalInfoContainer(clientId, operatorUid, requestId, RU);
        var message = getCommunicationMessage(clientId, operatorUid,
                Set.of(slotName), List.of(targetObjectId), requestId, RU)
                .stream()
                .filter(m -> m.getClientId().equals(clientId.asLong()))
                .filter(m -> m.getUserId().equals(operatorUid))
                .filter(m -> m.getEventId() == eventId)
                .filter(m -> m.getEventVersionId() == eventVersionId)
                .filter(m -> m.getSlot().getName().equals(slotName))
                .findAny();
        if (message.isEmpty()) {
            RecommendationLogger.logApplyFailed(
                    additionalInfo, eventId, eventVersionId, targetObjectId,
                    slot.getId(), NOT_ACTUAL, null
            );
            return false;
        }
        if (message
                .filter(m -> m.getMajorVersion() == majorDataVersion)
                .filter(m -> m.getMinorVersion() == minorDataVersion)
                .isEmpty()) {
            RecommendationLogger.logApplyFailed(
                    additionalInfo, eventId, eventVersionId, targetObjectId,
                    slot.getId(), WAS_CHANGED, message.get().getInventoryAdditionalData()
            );
            return false;
        }
        var button = message.get().getContent().getButtons().stream()
                .filter(b -> b.getAction().equals(EWebUIButtonActionType.BUTTON_ACTION_TYPE_ACTION) &&
                        b.getStyle().equals(EWebUIButtonStyle.BUTTON_STYLE_APPLY))
                .findAny();
        if (button.isEmpty()) {
            RecommendationLogger.logApplyFailed(
                    additionalInfo, eventId, eventVersionId, targetObjectId,
                    slot.getId(), NOT_SUPPORTED, message.get().getInventoryAdditionalData()
            );
            return false;
        }
        var eventVersion = Optional.ofNullable(eventVersionsRepository
                .getVersion(eventId, eventVersionId));
        var messageId = message.get().getMessageId();
        return processingFacade.confirmAction(
                additionalInfo,
                Map.of(messageId, new ActionTarget(targetObjectId, eventId, eventVersionId, majorDataVersion,
                        minorDataVersion, message.get().getInventoryAdditionalData())),
                Map.of(messageId, message.get()),
                eventVersion.isEmpty() ? emptyMap() : Map.of(messageId, eventVersion.get()),
                slot, button.get().getId())
                .getResult().equals(SummaryActionResult.SUCCESS);
    }

    public ActionResult<?> buttonAction(
            Long buttonId,
            Long requestId,
            ClientId clientId,
            Long operatorUid,
            String slotName,
            String lang,
            List<ActionTarget> targets
    ) {
        var slot = slotRepository.getSlotsByNames(List.of(slotName)).get(0);
        var additionalInfo = getAdditionalInfoContainer(clientId, operatorUid, requestId, lang);
        var targetByMessageId = listToMap(targets, t -> CommunicationHelper
                .calculateMessageId(operatorUid, clientId, t.getTargetObjectId(), t.getEventId()));
        var targetObjectIds = mapList(targets, ActionTarget::getTargetObjectId);
        var messageById = StreamEx.of(
                getCommunicationMessage(clientId, operatorUid, Set.of(slotName), targetObjectIds, requestId, lang))
                .filter(m -> m.getClientId().equals(clientId.asLong()))
                .filter(m -> m.getUserId().equals(operatorUid))
                .filter(m -> targetByMessageId.containsKey(m.getMessageId()))
                .filter(m -> m.getSlot().getName().equals(slotName))
                .toMap(CommunicationMessage::getMessageId, Function.identity());
        var eventIdToIterations = StreamEx.of(targets)
                .mapToEntry(ActionTarget::getEventId, ActionTarget::getEventVersionId)
                .distinct()
                .grouping();
        var eventVersions = StreamEx.of(
                eventVersionsRepository.getVersions(eventIdToIterations))
                .mapToEntry(CommunicationEventVersion::getEventId, Function.identity())
                .grouping(HashMap::new, Collectors.toMap(CommunicationEventVersion::getIter,
                        Function.identity()));
        var eventVersionByMessageId = EntryStream.of(targetByMessageId)
                .mapValues(t -> eventVersions
                        .getOrDefault(t.getEventId(), emptyMap())
                        .get(t.getEventVersionId()))
                .nonNullValues()
                .toMap();
        return processingFacade.confirmAction(
                additionalInfo,
                targetByMessageId,
                messageById,
                eventVersionByMessageId,
                slot, buttonId);
    }

    public Set<ClientId> getClientsWithRecommendationsForSlot(String slotName) {
        var slot = slotRepository.getSlotsByNames(List.of(slotName)).get(0);
        return StreamEx.of(inventoryClient.getClientsBySlot(slot.getId()).getClientIdsList())
                .map(ClientId::fromLong)
                .toSet();
    }

    /**
     * Проверяет наличие и актуальность сообщения и применяет действие кнопки.
     * @return
     *      null при неактуальном сообщении
     *      пустой список при неудачном выполнении действия
     *      список добавленных статусов при удачном выполнении действия
     */
    public Set<EMessageStatus> getCommunicationMessageAndHandleButton(
            AdditionalInfoContainer additionalInfo,
            String slotName,
            Long messageId,
            @Nullable Integer buttonId
    ) {
        var slotById = StreamEx.of(slotRepository.getSlotsByNames(Set.of(slotName)))
                .toMap(Slot::getId, Function.identity());
        var response = getCalculatedMessages(additionalInfo, List.of(),
                ETriggerType.on_action, slotById);
        var message = StreamEx.of(
                getCommunicationMessage(response, additionalInfo, slotById))
                .findAny(m -> m.getMessageId().equals(messageId))
                .orElse(null);
        if (message == null) {
            return null;
        }
        var version = eventVersionsRepository
                .getVersion(message.getEventId(), message.getEventVersionId());
        var target = new ActionTarget(message.getTargetObject().getId(),
                message.getEventId(), message.getEventVersionId(),
                message.getMajorVersion(), message.getMinorVersion(),
                message.getInventoryAdditionalData());
        var addedStatuses = Set.of(EMessageStatus.APPLY);
        if (buttonId == null) {
            buttonId = message.getContent()
                    .getButtons()
                    .stream()
                    .filter(b -> EWebUIButtonStyle.BUTTON_STYLE_CLOSE.equals(b.getStyle()))
                    .map(b -> (int) b.getId())
                    .findAny()
                    .orElse(null);
            addedStatuses = Set.of(EMessageStatus.REJECT);
        }
        if (buttonId == null) {
            return Set.of();
        }
        var result = processingFacade.confirmAction(
                additionalInfo,
                Map.of(messageId, target),
                Map.of(messageId, message),
                Map.of(messageId, version),
                slotById.values().iterator().next(),
                buttonId);
        if (SummaryActionResult.SUCCESS.equals(result.getResult())) {
            return addedStatuses;
        } else {
            return Set.of();
        }
    }

    public ActionResult<?> getCommunicationMessageAndApply(
            ClientId clientId,
            Long operatorUid,
            String slot,
            List<Long> targetObjectIds
    ) {
        var additionalInfo = getAdditionalInfoContainer(clientId, operatorUid, null, "ru");
        if (targetObjectIds.isEmpty()) {
            targetObjectIds = List.of(clientId.asLong());
        }
        var slotById = StreamEx.of(slotRepository.getSlotsByNames(List.of(slot)))
                .toMap(Slot::getId, Function.identity());
        var response = getCalculatedMessages(additionalInfo, targetObjectIds, ETriggerType.on_view, slotById);
        var messages = getCommunicationMessage(response, additionalInfo, slotById);

        var messageById = listToMap(messages, CommunicationMessage::getMessageId);
        var eventIdToIterations = StreamEx.of(messages)
                .mapToEntry(
                        CommunicationMessage::getEventId,
                        CommunicationMessage::getEventVersionId
                )
                .grouping();
        var eventVersions = StreamEx.of(
                eventVersionsRepository.getVersions(eventIdToIterations))
                .mapToEntry(CommunicationEventVersion::getEventId, Function.identity())
                .grouping(HashMap::new, Collectors.toMap(CommunicationEventVersion::getIter,
                        Function.identity()));
        var eventVersionByMessageId = EntryStream.of(messageById)
                .mapValues(m -> eventVersions
                        .getOrDefault(m.getEventId(), emptyMap())
                        .get(m.getEventVersionId()))
                .nonNullValues()
                .toMap();
        var targetByMessageId = EntryStream.of(messageById)
                .mapValues(m -> new ActionTarget(m.getTargetObject().getId(),
                        m.getEventId(), m.getEventVersionId(),
                        m.getMajorVersion(), m.getMinorVersion(),
                        m.getInventoryAdditionalData()))
                .toMap();
        return processingFacade.confirmAction(additionalInfo, targetByMessageId, messageById, eventVersionByMessageId,
                slotById.values().iterator().next(), 0L /* кнопка всегда одна для автоприменяемых */);
    }

    public List<CommunicationMessage> getCommunicationMessage(
            ClientId clientId,
            Long operatorUid,
            Set<String> slots,
            List<Long> targetObjectIds,
            String lang) {
        return getCommunicationMessage(clientId, operatorUid, slots, targetObjectIds, null, lang);
    }

    public AdditionalInfoContainer getAdditionalInfoContainer(
            ClientId clientId,
            @Nullable Long operatorUid,
            @Nullable Long parentRequestId,
            String lang
    ) {
        return new AdditionalInfoContainer()
                .withClientId(clientId)
                .withLanguage(lang)
                .withCookieAddedStatusesByMessageId(emptyMap())
                .withCookieRemovedStatusesByMessageId(emptyMap())
                .withCurrentTimeStamp(System.currentTimeMillis() / 1000)
                .withRequestId(parentRequestId != null ? parentRequestId : Trace.current().getTraceId())
                .withShardSupplier(info -> info.getClientId()
                        .map(clientIdOpt -> shardHelper.getShardByClientId(clientIdOpt))
                        .orElse(null))
                .withClientUidSupplier(info -> {
                    var shardOpt = info.getShard();
                    var clientIdOpt = info.getClientId();
                    return shardOpt.isEmpty() || clientIdOpt.isEmpty() ? null : clientRepository
                            .getClientData(shardOpt.get(), List.of(clientIdOpt.get()))
                            .stream().findAny()
                            .map(Client::getChiefUid)
                            .orElse(null);
                })
                .withNdsSupplier(info -> info.getClientId()
                        .map(clientNdsService::getClientNds)
                        .map(ClientNds::getNds)
                        .orElse(Percent.fromPercent(BigDecimal.ZERO)))
                .withUserIdSupplier(info -> operatorUid != null ? operatorUid : info.getClientUid().orElse(null))
                .withUserLoginSupplier(info -> info.getUserId()
                        .map(userId -> userRepository.getLoginsByUids(List.of(userId)))
                        .orElse(emptyMap())
                        .values().stream().findAny().orElse(null))
                .withCanUserWriteSupplier(info -> info.getUserId()
                        .map(userId -> info.getClientUid()
                                .map(clientUid -> rbacService.canWrite(userId, clientUid))
                                .orElse(false))
                        .orElse(null)
                )
                .withClientLoginSupplier(info -> info.getClientUid()
                        .map(userId -> userRepository.getLoginsByUids(List.of(userId)))
                        .orElse(emptyMap())
                        .values().stream().findAny().orElse(null))
                .withClientNameSupplier(info -> info.getClientUid()
                        .map(userId -> userRepository.getFiosByUids(info.getShard().get(), List.of(userId)))
                        .orElse(emptyMap())
                        .values().stream().findAny().orElse(null)
                )
                .withClientEmailSupplier(info -> info.getClientUid()
                        .map(userId -> userRepository.getUserEmail(info.getShard().get(), userId))
                        .orElse(null)
                )
                .withCurrencySupplier(info -> {
                    var shardOpt = info.getShard();
                    var clientIdOpt = info.getClientId();
                    return shardOpt.isEmpty() || clientIdOpt.isEmpty() ? null : clientRepository
                            .getClientData(shardOpt.get(), List.of(clientIdOpt.get()))
                            .stream().findAny()
                            .map(Client::getWorkCurrency)
                            .map(CurrencyCode::getCurrency)
                            .orElse(null);
                })
                .withTestIds(List.of())
                .withEnabledFeaturesSupplier(info -> info.getClientId()
                        .map(cl -> featureService.getEnabledForClientId(cl))
                        .orElse(emptySet())
                        .stream()
                        .filter(featureName -> featureName.contains("recommend") ||
                                RECOMMENDATION_FEATURE_NAMES.contains(featureName))
                        .collect(Collectors.toList())
                );
    }

    public List<CommunicationMessage> getCommunicationMessage(
            ClientId clientId,
            Long operatorUid,
            Set<String> slots,
            List<Long> targetObjectIds,
            @Nullable Long parentRequestId,
            String lang) {
        var additionalInfo = getAdditionalInfoContainer(clientId, operatorUid, parentRequestId, lang);
        return getCommunicationMessage(additionalInfo, slots, targetObjectIds, parentRequestId == null);
    }

    public List<CommunicationMessage> getCommunicationMessage(
            AdditionalInfoContainer additionalInfo,
            Set<String> slotNames,
            List<Long> targetObjectIds,
            boolean onView
    ) {
        var slots = slotRepository.getSlotsByNames(slotNames);
        var slotById = listToMap(slots, Slot::getId);
        if (targetObjectIds.isEmpty()) {
            targetObjectIds = getDefaultTargetObjectIds(slots, additionalInfo);
        }
        if (targetObjectIds.isEmpty()) {
            return emptyList();
        }
        var triggerType = onView ? ETriggerType.on_view : ETriggerType.on_action;
        var messages = getCalculatedMessages(additionalInfo, targetObjectIds, triggerType, slotById);
        return getCommunicationMessage(messages, additionalInfo, slotById);
    }

    public Response getCalculatedMessages(
            AdditionalInfoContainer additionalInfo,
            List<Long> targetObjectIds,
            ETriggerType triggerType,
            Map<Long, Slot> slotById) {
        if (targetObjectIds.isEmpty()) {
            targetObjectIds = getDefaultTargetObjectIds(slotById.values(), additionalInfo);
        }

        var shard = additionalInfo.getShard().orElse(null);
        var clientId = additionalInfo.getClientId().orElse(null);

        boolean sendAutobudgetRestartTimes = !ignoreAutobudgetRestartProp.getOrDefault(false) &&
                shard != null &&
                clientId != null &&
                slotById.values().stream()
                        .anyMatch(slot -> slot.getTargetObjectTypesList().contains(Slot.ESlotTargetObjectType.CAMPAIGN));

        List<CampStrategyRestartResultSuccess> actualRestartTimes = sendAutobudgetRestartTimes ?
                autobudgetRestartService.getActualRestartTimes(shard, getStrategyDto(shard, clientId, targetObjectIds))
                : emptyList();
        return getCalculatedMessages(additionalInfo, slotById.keySet(), targetObjectIds,
                triggerType, actualRestartTimes);
    }

    private List<Long> getDefaultTargetObjectIds(
            Collection<Slot> slots,
            AdditionalInfoContainer additionalInfo
    ) {
        var targetObjectIds = new ArrayList<Long>();
        var targetTypes = StreamEx.of(slots)
                .map(Slot::getTargetObjectTypesList)
                .flatMap(Collection::stream)
                .distinct()
                .toList();
        if (targetTypes.contains(Slot.ESlotTargetObjectType.CLIENT)) {
            targetObjectIds.add(additionalInfo.getClientId().get().asLong());
        }
        if (targetTypes.contains(Slot.ESlotTargetObjectType.USER) && additionalInfo.getUserId().isPresent()) {
            targetObjectIds.add(additionalInfo.getUserId().get());
        }
        if (targetTypes.contains(Slot.ESlotTargetObjectType.CAMPAIGN)) {
            var campaignIds = mapList(campaignRepository.getCampaigns(additionalInfo.getShard().get(),
                    new CampaignsSelectionCriteria()
                            .withClientIds(List.of(additionalInfo.getClientId().get()))
                            .withStatusArchived(false)
                            .withCampaignTypes(WEB_EDIT)),
                    Campaign::getId);
            targetObjectIds.addAll(campaignIds);
        }
        return targetObjectIds;
    }

    public List<CommunicationMessage> getCommunicationMessage(
            Response messages,
            AdditionalInfoContainer additionalInfo,
            Map<Long, Slot> slotById
    ) {
        var eventIdToIterations = StreamEx.of(messages.getSlotResponsesList())
                .flatCollection(SlotResponse::getObjectEventDataList)
                .mapToEntry(
                        ObjectEventData::getEventId,
                        ObjectEventData::getEventVersionId
                )
                .grouping();
        var eventVersions = StreamEx.of(
                eventVersionsRepository.getVersions(eventIdToIterations))
                .mapToEntry(CommunicationEventVersion::getEventId, Function.identity())
                .grouping(HashMap::new, Collectors.toMap(CommunicationEventVersion::getIter,
                        Function.identity()));

        var messageIds = StreamEx.of(messages.getSlotResponsesList())
                .flatCollection(SlotResponse::getObjectEventDataList)
                .map(ObjectEventData::getMessageId)
                .distinct()
                .toList();
        var caesarDataByMessageId = StreamEx.of(
                channelRepository.getCommunicationMessageByIds(messageIds))
                .toMap(CommunicationMessageData::getMessageId, Function.identity());
        return processingFacade.handle(messages, eventVersions, caesarDataByMessageId, slotById, additionalInfo);
    }

    public List<CommunicationMessageGroup> getCommunicationMessageGroup(
            ClientId clientId,
            Long operatorUid,
            Set<String> slots,
            List<Long> targetObjectIds,
            String interfaceLanguage) {
        var additionalInfo = getAdditionalInfoContainer(clientId, operatorUid, null, interfaceLanguage);
        var messages = getCommunicationMessage(additionalInfo, slots, targetObjectIds, true);

        var eventIdToIterations = StreamEx.of(messages)
                .mapToEntry(
                        CommunicationMessage::getEventId,
                        CommunicationMessage::getEventVersionId
                )
                .grouping();
        var eventVersions = StreamEx.of(
                eventVersionsRepository.getVersions(eventIdToIterations))
                .mapToEntry(CommunicationEventVersion::getEventId, Function.identity())
                .grouping(HashMap::new, Collectors.toMap(CommunicationEventVersion::getIter,
                        Function.identity()));
        var versionBySlotMessageId = StreamEx.of(messages)
                .mapToEntry(SlotMessageId::of)
                .invert()
                .mapValues(m -> eventVersions.get(m.getEventId()).get(m.getEventVersionId()))
                .toMap();

        return processingFacade.groupCommunicationMessages(messages, versionBySlotMessageId, additionalInfo);
    }

    private Map<Long, StrategyDto> getStrategyDto(
            int shard,
            ClientId clientId,
            List<Long> campaignIds
    ) {
        var campsWithCombinedGoals = metrikaCampaignRepository.getCampaignIdsWithCombinedGoals(shard, campaignIds);
        var selectionCriteria = new CampaignsSelectionCriteria()
                .withCampaignIds(campaignIds)
                .withClientIds(singletonList(clientId));

        return StreamEx.of(campaignRepository.getCampaigns(shard, selectionCriteria)).toMap(
                Campaign::getId,
                camp -> {
                    var strategy = camp.getStrategy();
                    var strategyData = strategy.getStrategyData();
                    var campOptionsStrategy = strategy.getStrategy();
                    var manualStrategy = Optional.ofNullable(CampOptionsStrategy.toSource(campOptionsStrategy))
                            .map(ru.yandex.direct.dbschema.ppc.enums.CampOptionsStrategy::getLiteral)
                            .orElse(null);
                    var platform = Optional.ofNullable(CampaignsPlatform.toSource(strategy.getPlatform()))
                            .map(ru.yandex.direct.dbschema.ppc.enums.CampaignsPlatform::getLiteral)
                            .orElse(null);
                    return new StrategyDto(
                            strategyData.getName(),
                            manualStrategy,
                            platform,
                            Optional.ofNullable(camp.getStartTime()).orElse(OLD_START_TIME),
                            null,
                            camp.getDayBudget(),
                            strategyData.getSum(),
                            Optional.ofNullable(camp.getOpts())
                                    .map(set -> set.contains(CampaignOpts.ENABLE_CPC_HOLD))
                                    .orElse(false),
                            Optional.ofNullable(camp.getTimeTarget())
                                    .map(TimeTarget::toRawFormat)
                                    .orElse(null),
                            camp.getStatusShow(),
                            Boolean.TRUE.equals(strategyData.getPayForConversion()),
                            strategyData.getGoalId(),
                            Optional.ofNullable(strategyData.getRoiCoef())
                                    .orElse(null),
                            strategyData.getLimitClicks(),
                            strategyData.getAvgCpm(),
                            strategyData.getAvgBid(),
                            strategyData.getAvgCpa(),
                            null,
                            null,
                            null,
                            campsWithCombinedGoals.contains(camp.getId()),
                            true,
                            null
                    );
                }
        );
    }

    public Map<Long, Map<DisabledField, Object>> getCampaignDisabledData(
            ClientId clientId,
            Set<String> slots,
            List<Long> targetObjectIds) {
        var actualSlots = slotRepository.getSlotsByNames(slots);
        var versionByAutoChangeableSettings = getVersionByAutoChangeableSettings(clientId, actualSlots, targetObjectIds);

        if (versionByAutoChangeableSettings.isEmpty()){
            return Collections.emptyMap();
        }

        var autoUpdatedFields = userActionLogReader.getLastAutoUpdatedSettings(
                clientId.asLong(), targetObjectIds, versionByAutoChangeableSettings.keySet(),
                maxTimeDepthForAutoApplyHistory.getOrDefault(12));

        return StreamEx.of(autoUpdatedFields)
                .filter(AutoUpdatedSettingsEvent::isAutoApplyEnabled)
                .mapToEntry(AutoUpdatedSettingsEvent::getTargetObjectId)
                .invert()
                .collapseKeys()
                .mapValues(list -> StreamEx.of(list)
                        .mapToEntry(versionByAutoChangeableSettings::get)
                        .mapValues(version -> DisabledField.CAMP_BUDGET_SUM)
                        // Когда станет более одного возможного значения, будем брать значение с версии
                        .invert()
                        .mapValues(v -> (Object) v)
                        .toMap()
                )
                .toMap();
    }

    public List<CommunicationMessage> getAutoAppliedRecommendation(
            ClientId clientId,
            Long operatorUid,
            Set<String> slots,
            List<Long> targetObjectIds,
            String lang) {
        /**
         * Алгоритм
         * 1) По slots находим рекомендации (версии) на эти слоты
         * 2) Откидываем выключенный для пользователя (по фиче и статусу)
         * 3) Для оставшихся рекомендаций достаем изменения для targetObjectIds соответствующих рекомендациям полей из истории изменений
         * 4) Оставляем только последнее изменение каждого поля каждого объекта
         * 5) Оставляем только автоприменения рекомендаций
         * 6) Ищем последнее включение опции автоприменения на кампаниях соответствующих targetObjectIds
         * 7) Из пункта 5 оставляем только изменения после последнего включения опции из пункта 6
         * 8) Для оставшихся изменений формируем сообщения
         * 8.1) Тексты берем из версии рекомендации
         * 8.2) Данные берем из истории изменений
         * 8.3) флаг isOption true - если сейчас опция на кампании выключена
         */
        var actualSlots = slotRepository.getSlotsByNames(slots);
        var versionByAutoChangeableSettings = getVersionByAutoChangeableSettings(clientId, actualSlots, targetObjectIds);

        if (versionByAutoChangeableSettings.isEmpty()){
            return Collections.emptyList();
        }

        var autoUpdatedFields = userActionLogReader.getLastAutoUpdatedSettings(
                clientId.asLong(), targetObjectIds, versionByAutoChangeableSettings.keySet(),
                maxTimeDepthForAutoApplyHistory.getOrDefault(12));
        var additionalInfo = getAdditionalInfoContainer(clientId, operatorUid, null, nvl(lang, "ru"));

        return StreamEx.of(autoUpdatedFields)
                .map(e -> {
                    var eventVersion = versionByAutoChangeableSettings.get(e.getSettings());
                    return processingFacade.formatAutoUpdatedSettingsEvent(e, eventVersion, actualSlots, additionalInfo);
                })
                .flatMap(Collection::stream)
                .toList();
    }

    public Map<ru.yandex.direct.useractionlog.model.AutoChangeableSettings, CommunicationEventVersion> getVersionByAutoChangeableSettings(
            ClientId clientId,
            List<Slot> slots,
            List<Long> targetObjectIds
    ) {
        if (CollectionUtils.isEmpty(targetObjectIds)) {
            return Collections.emptyMap();
        }

        var clientFeatures = featureService.getEnabledForClientId(clientId);
        var requestedSlotIds = slots
                .stream()
                .map(Slot::getId)
                .collect(Collectors.toList());

        return eventVersionsRepository
                .getVersionsByStatuses(List.of(CommunicationEventVersionStatus.ACTIVE))
                .stream()
                .filter(v -> isSlotApplicable(requestedSlotIds, v.getSlots()))
                .filter(v -> Objects.nonNull(v.getAutoChangeableSettings()))
                .filter(v -> CollectionUtils.isEmpty(v.getFeatures()) || clientFeatures.containsAll(v.getFeatures()))
                .collect(Collectors.toMap(e -> toUserActionModel(e.getAutoChangeableSettings()), e -> e));
    }

        /**
         * Применима ли данная рекомендация с учетом запрошенных слотов
         * @param requestedSlots запрошенные фронтом слоты
         * @param recommendationSlots слоты указанные в рекомендации
         */
    private boolean isSlotApplicable(List<Long> requestedSlots, List<Long> recommendationSlots){
        return CollectionUtils.isEmpty(requestedSlots) || CollectionUtils.isEmpty(recommendationSlots)
                || recommendationSlots.stream().anyMatch(requestedSlots::contains);
    }

    /**
     * Перекладываем настройки в модель понятную для user-action-log/common. Делаем это исключительно для того
     * чтобы из-за одного класса не добавлять лишнюю зависимость из alw на core и наоборот.
     */
    private ru.yandex.direct.useractionlog.model.AutoChangeableSettings toUserActionModel(AutoChangeableSettings settings){
        if (settings == null){
            return null;
        }

        return new ru.yandex.direct.useractionlog.model.AutoChangeableSettings()
                .withItem(settings.getItem())
                .withSubitem(settings.getSubitem())
                .withType(settings.getType())
                .withRecommendationOptionName(settings.getRecommendationOptionName());
    }
}
