package ru.yandex.direct.communication.service;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.communication.container.web.CommunicationMessage;
import ru.yandex.direct.communication.facade.ActionResult;
import ru.yandex.direct.communication.facade.CommunicationEventVersionProcessingFacade;
import ru.yandex.direct.communication.model.Slot;
import ru.yandex.direct.communication.model.inventory.Response;
import ru.yandex.direct.communication.model.inventory.SlotResponse;
import ru.yandex.direct.communication.repository.CommunicationSlotRepository;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersion;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersionStatus;
import ru.yandex.direct.core.entity.communication.repository.CommunicationEventVersionsRepository;
import ru.yandex.direct.dbutil.model.ClientId;

import static java.util.Collections.emptyMap;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class CommunicationEventService {

    private static final Logger logger = LoggerFactory.getLogger(CommunicationEventService.class);

    public static final String AUTO_APPLY_PRICE_SLOT = "auto_apply_price";

    private final CommunicationEventVersionsRepository communicationEventVersionsRepository;
    private final CommunicationSlotRepository slotRepository;
    private final CommunicationEventVersionProcessingFacade processingFacade;
    private final CommunicationChannelService channelService;

    @Autowired
    public CommunicationEventService(
            CommunicationEventVersionsRepository communicationEventVersionsRepository,
            CommunicationSlotRepository slotRepository,
            CommunicationEventVersionProcessingFacade processingFacade,
            CommunicationChannelService channelService
    ) {
        this.communicationEventVersionsRepository = communicationEventVersionsRepository;
        this.slotRepository = slotRepository;
        this.processingFacade = processingFacade;
        this.channelService = channelService;
    }

    /**
     * Находим все события которые должны быть добавлены сразу после создания клиента
     *
     * @param clientId идентификатор клиента для которого необходимо выполнить действия после создания
     */
    public void processOnClientCreateEvents(ClientId clientId) {
        var versionsWithOnClientAction =
                communicationEventVersionsRepository
                        .getVersionsByStatuses(List.of(CommunicationEventVersionStatus.ACTIVE))
                        .stream()
                        .filter((e) -> !isBlank(e.getOnClientCreate()))
                        .collect(toList());
        logger.info("{} communication event versions were found for clientId {}",
                versionsWithOnClientAction.size(),
                clientId);

        var additionalInfo = channelService
                .getAdditionalInfoContainer(clientId, null, null, "ru");
        versionsWithOnClientAction
                .stream()
                .collect(groupingBy(CommunicationEventVersion::getOnClientCreate))
                .forEach((k, v) -> processingFacade.processOnClientCreateEvents(additionalInfo, k, v));
    }

    /**
     * Запускаем процессы коммуникации стартующие после создания заявки на МСС
     *
     * @param controlClientId id управляющего аккаунта МСС
     * @param managedClientId id клиента, который подал заявку на управление
     */
    public void processOnClientMccRequest(ClientId controlClientId, ClientId managedClientId) {
        try {
            var versions =
                    communicationEventVersionsRepository
                            .getVersionsByStatuses(List.of(CommunicationEventVersionStatus.ACTIVE))
                            .stream()
                            .filter((e) -> !isBlank(e.getOnClientMccRequest()))
                            .collect(toList());
            var additionalInfo = channelService
                    .getAdditionalInfoContainer(controlClientId, null, null, "ru")
                    .withAbstractInfo(new HashMap(Map.of("managedClientId", managedClientId)))
                    .withCookieAddedStatusesByMessageId(emptyMap())
                    .withCookieRemovedStatusesByMessageId(emptyMap());
            versions.stream()
                    .collect(groupingBy(CommunicationEventVersion::getOnClientMccRequest))
                    .forEach((k, v) -> processingFacade.processOnTrigger(k, v, additionalInfo));
            Response response = (Response) additionalInfo.getAbstractInfo().get().get("response");
            if (response == null) {
                return;
            }
            var slotById = listToMap(
                    slotRepository.getSlotsByIds(
                            mapList(response.getSlotResponsesList(), SlotResponse::getSlotId)),
                    Slot::getId
            );
            var messages = channelService.getCommunicationMessage(
                    response,
                    additionalInfo,
                    slotById
            );
            var versionByIdAndIter = listToMap(versions, v -> Pair.of(v.getEventId(), v.getIter()));
            var versionByMessageId = listToMap(messages, CommunicationMessage::getMessageId,
                    m -> versionByIdAndIter.get(Pair.of(m.getEventId(), m.getEventVersionId())));
            processingFacade.sendToChannel("MAIL", messages, additionalInfo, versionByMessageId);
        } catch (RuntimeException ex) {
            logger.error("processOnClientMccRequest failed", ex);
        }
    }

    /**
     * Запускаем процессы коммуникации стартующие после автопополнения кешбеком
     *
     * @param paramsByClient входные данные для каждого клиента
     */
    public void processOnClientCashback(Map<ClientId, Map<String, String>> paramsByClient) {
        String trigger = "onClientCashback";
        var processorByVersion = StreamEx.of(communicationEventVersionsRepository
                .getVersionsByStatuses(List.of(CommunicationEventVersionStatus.ACTIVE)))
                .mapToEntry(e -> nvl(e.getProcessorByTrigger(), Collections.<String, String>emptyMap())
                        .get(trigger))
                .nonNullValues()
                .toMap();
        paramsByClient.forEach((clientId, params) -> processOnClientCashback(clientId, params, processorByVersion));
    }

    private void processOnClientCashback(
            ClientId clientId, Map<String, String> params,
            Map<CommunicationEventVersion, String> processorByVersion
    ) {
        var additionalInfo = channelService
                .getAdditionalInfoContainer(clientId, null, null, "ru")
                .withAbstractInfo(new HashMap(Map.of("params", params)))
                .withCookieAddedStatusesByMessageId(emptyMap())
                .withCookieRemovedStatusesByMessageId(emptyMap());
        EntryStream.of(processorByVersion)
                .invert()
                .grouping()
                .forEach((k, v) -> processingFacade.processOnTrigger(k, v, additionalInfo));
        Response response = (Response) additionalInfo.getAbstractInfo().get().get("response");
        if (response == null) {
            return;
        }
        var slotById = listToMap(
                slotRepository.getSlotsByIds(
                        mapList(response.getSlotResponsesList(), SlotResponse::getSlotId)),
                Slot::getId
        );
        var messages = channelService.getCommunicationMessage(
                response,
                additionalInfo,
                slotById
        );
        var versionByIdAndIter = listToMap(processorByVersion.keySet(), v -> Pair.of(v.getEventId(), v.getIter()));
        var versionByMessageId = listToMap(messages, CommunicationMessage::getMessageId,
                m -> versionByIdAndIter.get(Pair.of(m.getEventId(), m.getEventVersionId())));
        processingFacade.sendToChannel("MAIL", messages, additionalInfo, versionByMessageId);
    }

    /**
     * Берем автоприменяемые рекомендации для этих кампаний и применяем их.
     */
    public ActionResult processAutoApplyOnClient(ClientId clientId, List<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            logger.debug("There is not campaigns with autoapplying option on client " + clientId);
            return ActionResult.createEmpty();
        }
        return channelService.getCommunicationMessageAndApply(clientId, null, AUTO_APPLY_PRICE_SLOT, campaignIds);
    }

    /**
     * @return список клиентов, для которых необходимо проверить наличие рекомендаций на автоприменение
     */
    public Collection<ClientId> getClientWithAutoApplyRecommendations() {
        return channelService.getClientsWithRecommendationsForSlot(AUTO_APPLY_PRICE_SLOT);
    }
}
