package ru.yandex.direct.libs.curator;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Duration;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.PreDestroy;

import org.apache.curator.RetryPolicy;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.CuratorFrameworkFactory;
import org.apache.curator.framework.imps.CuratorFrameworkState;
import org.apache.curator.framework.recipes.locks.InterProcessLock;
import org.apache.curator.framework.recipes.locks.InterProcessMultiLock;
import org.apache.curator.framework.recipes.locks.InterProcessSemaphoreMutex;
import org.apache.curator.retry.ExponentialBackoffRetry;

import ru.yandex.direct.libs.curator.lock.CuratorLock;
import ru.yandex.direct.libs.curator.lock.CuratorLockTimeoutException;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Обертка над CuratorFrameworkFactory
 * Строку подключения к ZooKeeper берет из свойств, политика перезапросов определена внутри
 * Хранит внутри один запущенный инстанс для общих нужд, но может предоставить и новый.
 */
@ParametersAreNonnullByDefault
public class CuratorFrameworkProvider {
    private static final RetryPolicy DEFAULT_RETRY_POLICY = new ExponentialBackoffRetry(1_000, 3, 60_000);

    private final String zkHosts;
    private final Path lockPath;
    private final RetryPolicy defaultRetryPolicy;

    private CuratorFramework curatorFramework;

    public CuratorFrameworkProvider(String connectString, String lockPath) {
        this(connectString, lockPath, DEFAULT_RETRY_POLICY);
    }

    public CuratorFrameworkProvider(String connectString, String lockPath, RetryPolicy defaultRetryPolicy) {
        this.lockPath = Paths.get(lockPath).normalize();
        checkArgument(this.lockPath.isAbsolute(), "Lock path should be absolute");

        this.defaultRetryPolicy = defaultRetryPolicy;
        this.zkHosts = connectString;
    }

    public String getZkHosts() {
        return zkHosts;
    }

    public Path getLockPath() {
        return lockPath;
    }

    /**
     * Получить новый запущенный {@link CuratorFramework}. Вероятнее всего, вам не нужен отдельный новый инстанс и
     * достаточно общего - пользуйтесь {@link #getDefaultCurator()}
     *
     * @return client
     */
    public CuratorFramework getNewCurator() {
        CuratorFramework client = CuratorFrameworkFactory.newClient(zkHosts, defaultRetryPolicy);
        try {
            client.start();
            return client;
        } catch (Exception e) {
            client.close();
            throw e;
        }
    }

    /**
     * Получить общий запущенный {@link CuratorFramework}
     *
     * @return client
     */
    public synchronized CuratorFramework getDefaultCurator() {
        if (curatorFramework == null || curatorFramework.getState() == CuratorFrameworkState.STOPPED) {
            curatorFramework = getNewCurator();
        }
        return curatorFramework;
    }

    /**
     * Получить обертку над запущенным {@link CuratorFramework}
     *
     * @return новый инстанс {@link CuratorWrapper} поверх общего инстанса {@link CuratorFramework}
     */
    public CuratorWrapper getDefaultWrapper() {
        return new CuratorWrapper(getDefaultCurator());
    }

    /**
     * Вернуть acquired лок c заданным базовым именем и набором суффиксов
     */
    public CuratorLock getLock(String baseLockName, Collection<String> suffixes, Duration timeout,
                               @Nullable Runnable onLoss) {
        return getLock(mapList(suffixes, o -> String.format("%s-%s", baseLockName, o)), timeout, onLoss);
    }

    /**
     * Вернуть acquired лок c заданным базовым именем
     */
    public CuratorLock getLock(String lockName, Duration timeout, @Nullable Runnable onLoss) {
        return getLock(List.of(lockName), timeout, onLoss);
    }

    /**
     * Вернуть acquired лок, объединяющий в себе несколько локов с заданными именами
     *
     * @param lockNames имена локов для получения
     * @param timeout   время ожидания лока
     * @param onLoss    коллбек, который будет вызван при потере лока
     * @throws CuratorLockTimeoutException - не смогли взять lock из-за таймаута
     * @throws CuratorRuntimeException     - какая-то другая причина
     */
    public CuratorLock getLock(Collection<String> lockNames, Duration timeout, @Nullable Runnable onLoss) {
        checkArgument(!lockNames.isEmpty(), "There must be some lock names");
        for (String lockName : lockNames) {
            checkArgument(CuratorLock.isValidLockName(lockName), String.format("Invalid name for lock: %s", lockName));
        }

        CuratorLock lock = createLock(lockNames, onLoss);
        // Получаем лок
        boolean locked;
        try {
            locked = lock.acquire(timeout.toMillis(), TimeUnit.MILLISECONDS);
        } catch (Exception e) {
            throw new CuratorRuntimeException(
                    String.format("Cannot acquire locks %s for timeout %s", lockNames, timeout), e);
        }

        if (!locked) {
            throw new CuratorLockTimeoutException(
                    String.format("Cannot acquire locks %s for timeout %s", lockNames, timeout));
        } else {
            return lock;
        }
    }

    /**
     * Вернуть лок, объединяющий в себе несколько локов с заданными именами
     *
     * @param lockName имея лока для получения
     * @param onLoss   коллбек, который будет вызван при потере лока
     */
    public CuratorLock createLock(String lockName, @Nullable Runnable onLoss) {
        return createLock(List.of(lockName), onLoss);
    }

    /**
     * Вернуть лок, объединяющий в себе несколько локов с заданными именами
     *
     * @param lockNames имена локов для получения
     * @param onLoss    коллбек, который будет вызван при потере лока
     */
    public CuratorLock createLock(Collection<String> lockNames, @Nullable Runnable onLoss) {
        checkArgument(!lockNames.isEmpty(), "There must be some lock names");
        for (String lockName : lockNames) {
            checkArgument(CuratorLock.isValidLockName(lockName), String.format("Invalid name for lock: %s", lockName));
        }

        // Создаем лок
        List<String> lockPaths = mapList(lockNames, n -> lockPath.resolve(n).toString());
        CuratorFramework framework = getDefaultCurator();
        InterProcessLock innerLock;
        if (lockPaths.size() > 1) {
            innerLock = new InterProcessMultiLock(framework, lockPaths);
        } else {
            innerLock = new InterProcessSemaphoreMutex(framework, lockPath.resolve(lockPaths.get(0)).toString());
        }
        CuratorLock lock = new CuratorLock(innerLock, onLoss);
        framework.getConnectionStateListenable().addListener(lock);
        return lock;
    }

    @PreDestroy
    public void closeDefault() {
        if (curatorFramework != null) {
            curatorFramework.close();
        }
    }
}
