package ru.yandex.direct.libs.curator.lock;

import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.regex.Pattern;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.recipes.locks.InterProcessLock;
import org.apache.curator.framework.state.ConnectionState;
import org.apache.curator.framework.state.ConnectionStateListener;

/**
 * Простой зукиперный лок, который умеет в try с ресурсами и обрабатывать состояние соединения
 */
@ParametersAreNonnullByDefault
public class CuratorLock implements InterProcessLock, AutoCloseable, ConnectionStateListener {
    private static final Pattern CORRECT_NAME_PATTERN = Pattern.compile("[-A-Za-z0-9_]+");

    private final InterProcessLock internalLock;
    private final Runnable connectionLossCallback;
    private final AtomicBoolean callbackRunned;

    public CuratorLock(InterProcessLock internalLock, @Nullable Runnable connectionLossCallback) {
        this.internalLock = internalLock;
        this.connectionLossCallback = connectionLossCallback;
        callbackRunned = new AtomicBoolean(false);
    }

    @Override
    public void close() throws Exception {
        release();
    }

    @Override
    public void acquire() throws Exception {
        internalLock.acquire();
    }

    @Override
    public boolean acquire(long time, TimeUnit unit) throws Exception {
        return internalLock.acquire(time, unit);
    }

    @Override
    public void release() throws Exception {
        internalLock.release();
    }

    @Override
    public boolean isAcquiredInThisProcess() {
        return internalLock.isAcquiredInThisProcess();
    }

    /**
     * Проверить, что строка может быть использована в качестве имени лока
     */
    public static boolean isValidLockName(String name) {
        return CORRECT_NAME_PATTERN.matcher(name).matches();
    }

    @Override
    public void stateChanged(CuratorFramework client, ConnectionState newState) {
        if (!isAcquiredInThisProcess() || newState != ConnectionState.LOST) {
            return;
        }

        if (connectionLossCallback != null && !callbackRunned.getAndSet(true)) {
            connectionLossCallback.run();
        }
    }
}
