package ru.yandex.direct.currency;

import java.math.BigDecimal;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;

import ru.yandex.direct.currency.currencies.CurrencyByn;
import ru.yandex.direct.currency.currencies.CurrencyChf;
import ru.yandex.direct.currency.currencies.CurrencyEur;
import ru.yandex.direct.currency.currencies.CurrencyGbp;
import ru.yandex.direct.currency.currencies.CurrencyKzt;
import ru.yandex.direct.currency.currencies.CurrencyRub;
import ru.yandex.direct.currency.currencies.CurrencyTry;
import ru.yandex.direct.currency.currencies.CurrencyUah;
import ru.yandex.direct.currency.currencies.CurrencyUsd;
import ru.yandex.direct.currency.currencies.CurrencyYndFixed;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.function.Function.identity;

/**
 * Содержит в себе список валют, с которыми умеет работать Direct.
 */
@ParametersAreNonnullByDefault
public class Currencies {
    public static final BigDecimal EPSILON = BigDecimal.valueOf(1, 7);

    private static final ImmutableMap<CurrencyCode, Currency> ALL = new ImmutableMap.Builder<CurrencyCode, Currency>()
            .put(CurrencyCode.YND_FIXED, CurrencyYndFixed.getInstance())
            .put(CurrencyCode.RUB, CurrencyRub.getInstance())
            .put(CurrencyCode.UAH, CurrencyUah.getInstance())
            .put(CurrencyCode.USD, CurrencyUsd.getInstance())
            .put(CurrencyCode.EUR, CurrencyEur.getInstance())
            .put(CurrencyCode.KZT, CurrencyKzt.getInstance())
            .put(CurrencyCode.CHF, CurrencyChf.getInstance())
            .put(CurrencyCode.TRY, CurrencyTry.getInstance())
            .put(CurrencyCode.BYN, CurrencyByn.getInstance())
            .put(CurrencyCode.GBP, CurrencyGbp.getInstance())
            .build();

    private static final ImmutableMap<String, Currency> CURRENCIES = ImmutableMap.copyOf(
            ALL.entrySet().stream().collect(Collectors.toMap(e -> e.getKey().name(), Map.Entry::getValue)));

    private static final ImmutableMap<Integer, Currency> ALL_BY_ISO_CODE =
            ImmutableMap.copyOf(CURRENCIES.values().stream()
                    .filter(c -> c.getIsoNumCode() != null)
                    .collect(Collectors.toMap(Currency::getIsoNumCode, identity())));

    public static ImmutableMap<String, Currency> getCurrencies() {
        return CURRENCIES;
    }

    /**
     * Получить валюту по ее {@link CurrencyCode}
     *
     * @param code код валюты
     * @return свойства валюты
     * @throws NullPointerException если такой валюты нет в Директе
     */
    public static Currency getCurrency(CurrencyCode code) {
        return checkNotNull(ALL.get(code), "Unknown currency %s", code);
    }

    /**
     * Получить валюту по ее строковому коду валюты
     *
     * @param code строка с кодом валюты
     * @throws NullPointerException если такой валюты нет в Директе
     */
    public static Currency getCurrency(String code) {
        return checkNotNull(CURRENCIES.get(code), "Unknown currency %s", code);
    }

    /**
     * Получить валюту по ее ISO коду
     *
     * @param isoCode ISO код валюты
     * @throws NullPointerException если такой валюты нет в Директе
     */
    public static Currency getCurrency(Integer isoCode) {
        return checkNotNull(ALL_BY_ISO_CODE.get(isoCode), "Unknown currency for iso code %s", isoCode);
    }

    public static boolean currencyExists(String code) {
        return CURRENCIES.containsKey(code);
    }
}
