package ru.yandex.direct.currency;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.Locale;
import java.util.Objects;

import ru.yandex.direct.i18n.Translatable;
import ru.yandex.direct.i18n.Translator;

/**
 * Реализация {@link Translatable}, которая позволяет перевести сообщение с суммой денег.
 */
public class CurrencyAmount implements Translatable {

    private final Currency currency;
    private final BigDecimal amount;

    private CurrencyAmount(BigDecimal amount, Currency currency) {
        this.currency = currency;
        this.amount = amount;
    }

    /**
     * @return {@link CurrencyAmount} &ndash; представление денег с переводом.
     */
    public static CurrencyAmount fromMoney(Money money) {
        return new CurrencyAmount(money.bigDecimalValue(), money.getCurrencyCode().getCurrency());
    }

    @Override
    public String translate(Translator translator) {
        // повторена логика из тестов, но она нуждается в серьёзном переосмыслении
        DecimalFormatSymbols formatSymbols = DecimalFormatSymbols.getInstance(Locale.forLanguageTag("en"));
        // для разделения групп используем неразрывный пробел
        formatSymbols.setGroupingSeparator(' ');
        DecimalFormat decimalFormat = new DecimalFormat("#,##0.00");
        decimalFormat.setDecimalFormatSymbols(formatSymbols);

        Translatable t = currency.getCode().getTranslation().shortForm();
        String currencyTranslation = t.translate(translator);

        return String.format("%s %s", decimalFormat.format(amount.doubleValue()), currencyTranslation);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        CurrencyAmount that = (CurrencyAmount) o;
        return Objects.equals(currency, that.currency) &&
                Objects.equals(amount, that.amount);
    }

    @Override
    public int hashCode() {
        return Objects.hash(currency, amount);
    }
}
