package ru.yandex.direct.currency;

import java.util.Arrays;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonCreator;
import org.apache.commons.lang3.EnumUtils;

import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toMap;

/**
 * Enum для типа валюты.
 */
public enum CurrencyCode {
    YND_FIXED,
    RUB,
    UAH,
    USD,
    EUR,
    KZT,
    CHF,
    TRY,
    BYN,
    GBP;

    private static final Map<String, CurrencyCode> CODES_BY_NAME = Arrays.stream(CurrencyCode.values())
            .collect(toMap(Enum::name, identity()));

    public static final Set<CurrencyCode> NOT_FOR_CLIENT_CREATION = Set.of(YND_FIXED);

    public static final Set<CurrencyCode> QUASICURRENCY_CODES = Set.of(BYN, KZT);

    public static final Set<CurrencyCode> AUTO_OVERDRAFT_AVAILABLE_CURRENCIES = Set.of(RUB, BYN, KZT);

    public static final Set<CurrencyCode> DONT_SHOW_WALLET_NDS_IN_INTERFACE =
            Set.of(BYN, USD, EUR, TRY, YND_FIXED, GBP);

    public static final Set<CurrencyCode> CASHBACK_AVAILABLE_CURRENCIES = Set.of(RUB, KZT, BYN);

    /**
     * Получить валютные константы и параметры
     *
     * @return синглтон {@link Currency} соответствующий коду данной валюты
     * @throws NullPointerException если для данного кода валюты параметры неизвестны
     */
    public Currency getCurrency() {
        return Currencies.getCurrency(this);
    }

    public CurrencyTranslation getTranslation() {
        return CurrencyTranslationHolder.ofCurrency(this);
    }

    /**
     * Возвращает истину, если переданная строка является кодом реальной валюты, с которой работает Директ
     */
    public static boolean isRealCurrencyCode(String code) {
        return EnumUtils.isValidEnum(CurrencyCode.class, code) && !YND_FIXED.name().equals(code);
    }

    @JsonCreator
    @Nullable
    public static CurrencyCode parse(String code) {
        return CODES_BY_NAME.get(code.toUpperCase());
    }
}
