package ru.yandex.direct.currency;

import java.util.Map;

import com.google.common.collect.ImmutableMap;

import ru.yandex.direct.i18n.Translatable;

/**
 * Содержит соответствие валюты и её текстового представления
 */
public class CurrencyTranslationHolder {
    private static final Map<CurrencyCode, CurrencyTranslation> CURRENCY_PRESENTATION_BY_CODE =
            initCurrencyPresentationByCurrencyCode();

    private CurrencyTranslationHolder() {
    }

    /**
     * Метод возвращает перевод для конкретной валюты
     *
     * @param currencyCode код валюты
     * @throws IllegalArgumentException если перевод для данного кода валюты не найден
     */
    public static CurrencyTranslation ofCurrency(CurrencyCode currencyCode) {
        CurrencyTranslation currencyTranslation = CURRENCY_PRESENTATION_BY_CODE.get(currencyCode);
        if (currencyTranslation == null) {
            throw new IllegalArgumentException("There is no translation for this currency code: " + currencyCode);
        }
        return currencyTranslation;
    }

    private static Map<CurrencyCode, CurrencyTranslation> initCurrencyPresentationByCurrencyCode() {
        CurrenciesTranslations t = translations();
        return ImmutableMap.<CurrencyCode, CurrencyTranslation>builder()
                .put(CurrencyCode.YND_FIXED, translation(t.yndShort(), t.yndFull(), t.yndSymbol()))
                .put(CurrencyCode.RUB, translation(t.rubShort(), t.rubFull(), t.rubSymbol()))
                .put(CurrencyCode.UAH, translation(t.uahShort(), t.uahFull(), t.uahSymbol()))
                .put(CurrencyCode.KZT, translation(t.kztShort(), t.kztFull(), t.kztSymbol()))
                .put(CurrencyCode.USD, translation(t.usdShort(), t.usdFull(), t.usdSymbol()))
                .put(CurrencyCode.EUR, translation(t.eurShort(), t.eurFull(), t.eurSymbol()))
                .put(CurrencyCode.CHF, translation(t.chfShort(), t.chfFull(), t.chfSymbol()))
                .put(CurrencyCode.TRY, translation(t.tryShort(), t.tryFull(), t.trySymbol()))
                .put(CurrencyCode.BYN, translation(t.bynShort(), t.bynFull(), t.bynSymbol()))
                .put(CurrencyCode.GBP, translation(t.gbpShort(), t.gbpFull(), t.gbpSymbol()))
                .build();
    }

    private static CurrencyTranslation translation(Translatable shortValue, Translatable longValue, Translatable symbol) {
        return new CurrencyTranslation(shortValue, longValue, symbol);
    }

    private static CurrenciesTranslations translations() {
        return CurrenciesTranslations.INSTANCE;
    }
}
