package ru.yandex.direct.currency;

import java.math.BigDecimal;
import java.math.MathContext;
import java.math.RoundingMode;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.currency.MoneyUtils.ROUND_CORRECTION;
import static ru.yandex.direct.currency.MoneyUtils.ROUND_CORRECTION_SCALE;
import static ru.yandex.direct.currency.MoneyUtils.checkMoneyCurrencyCodesAreSame;

/**
 * Позволяет связать сумму и валюту
 */
@ParametersAreNonnullByDefault
public class Money implements Comparable<Money> {

    public static final MathContext MONEY_MATH_CONTEXT = MathContext.DECIMAL128;
    @SuppressWarnings("WeakerAccess")
    public static final int MICRO_MULTIPLIER_SCALE = 6;
    /**
     * Количество знаков после запятой для округления до минимальной суммы денег (копеек, центов и т.п.)
     */
    @SuppressWarnings("WeakerAccess")
    public static final int MONEY_CENT_SCALE = 2;
    public static final BigDecimal MICRO_MULTIPLIER = BigDecimal.valueOf(1, -MICRO_MULTIPLIER_SCALE);

    private final BigDecimal value;
    private final CurrencyCode currencyCode;

    private Money(BigDecimal value, CurrencyCode currencyCode) {
        this.currencyCode = currencyCode;
        this.value = value;
    }

    public static Money valueOf(BigDecimal value, CurrencyCode currencyCode) {
        checkNotNull(value, "Can't create money from null");
        return new Money(value, currencyCode);
    }

    /**
     * @return {@link Money}, соответствующий переданным значениям {@code value} и {@code currencyCode}
     */
    public static Money valueOf(double value, CurrencyCode currencyCode) {
        return valueOf(
                (new BigDecimal(Math.round(value * ROUND_CORRECTION)))
                        .scaleByPowerOfTen(-ROUND_CORRECTION_SCALE), currencyCode
        );
    }

    /**
     * @return {@link Money}, соответствующий переданным значениям {@code value} и {@code currencyCode}
     */
    public static Money valueOf(String value, CurrencyCode currencyCode) {
        return valueOf(new BigDecimal(value, MONEY_MATH_CONTEXT), currencyCode);
    }

    /**
     * @return {@link Money}, соответствующий переданным значениям {@code value} и {@code currencyCode}
     */
    public static Money valueOf(String value, String currencyCode) {
        return valueOf(value, CurrencyCode.valueOf(currencyCode));
    }

    /**
     * Позволяет получить экземпляр {@link Money}, если известно количество микроденег.
     * <p>
     * Микроденьги используются в API Директа, Показометра, Торгов БК
     */
    public static Money valueOfMicros(long value, CurrencyCode currencyCode) {
        return valueOf(BigDecimal.valueOf(value, MICRO_MULTIPLIER_SCALE), currencyCode);
    }

    /**
     * Для указанной валюты возвращает минимальную недостижимую ставку. По сути {@code maxBid + auctionStep}
     */
    public static Money unreachableBid(Currency currency) {
        BigDecimal maxPrice = currency.getMaxPrice();
        BigDecimal unreachableBid = maxPrice.add(currency.getAuctionStep());
        return valueOf(unreachableBid, currency.getCode());
    }

    public BigDecimal bigDecimalValue() {
        return value;
    }

    public long micros() {
        return value.scaleByPowerOfTen(MICRO_MULTIPLIER_SCALE).longValue();
    }

    public CurrencyCode getCurrencyCode() {
        return currencyCode;
    }

    @Override
    public int compareTo(Money other) {
        checkMoneyCurrencyCodesAreSame(this, other);
        return this.value.compareTo(other.value);
    }

    public boolean lessThan(Money other) {
        return this.compareTo(other) < 0;
    }

    public boolean greaterThan(Money other) {
        return this.compareTo(other) > 0;
    }

    public boolean lessThanOrEqual(Money other) {
        return this.compareTo(other) <= 0;
    }

    /**
     * @return {@code true}, если хранимая величина больше нуля. Иначе {@code false}
     */
    public boolean greaterThanZero() {
        return value.compareTo(BigDecimal.ZERO) > 0;
    }

    /**
     * @return {@code true}, если хранимая величина меньше нуля. Иначе {@code false}
     */
    public boolean lessThanZero() {
        return value.compareTo(BigDecimal.ZERO) < 0;
    }

    /**
     * @return {@code true}, если хранимая величина равна нулю. Иначе {@code false}
     */
    public boolean isZero() {
        return value.compareTo(BigDecimal.ZERO) == 0;
    }

    /**
     * @return {@code true}, если хранимая величина большие или равна {@link Currencies#EPSILON}. Иначе {@code false}
     */
    public boolean greaterThanOrEqualEpsilon() {
        return value.compareTo(Currencies.EPSILON) >= 0;
    }

    /**
     * @return {@code true}, если хранимая величина меньше или равна {@link Currencies#EPSILON}. Иначе {@code false}
     */
    public boolean lessThanOrEqualEpsilon() {
        return value.compareTo(Currencies.EPSILON) <= 0;
    }

    public Money add(Money other) {
        checkArgument(other.currencyCode == currencyCode, "Can't calculate sum of money in different currencies");
        return Money.valueOf(value.add(other.value, MONEY_MATH_CONTEXT), currencyCode);
    }

    public Money add(Percent percent) {
        return Money.valueOf(value.add(value.multiply(percent.asRatio()), MONEY_MATH_CONTEXT), currencyCode);
    }

    public Money subtract(Money other) {
        checkArgument(other.currencyCode == currencyCode, "Can't calculate sum of money in different currencies");
        return Money.valueOf(value.subtract(other.value, MONEY_MATH_CONTEXT), currencyCode);
    }

    public Money subtract(Percent percent) {
        return Money.valueOf(value.subtract(value.multiply(percent.asRatio()), MONEY_MATH_CONTEXT), currencyCode);
    }

    public Money multiply(BigDecimal multiplier) {
        return Money.valueOf(value.multiply(multiplier, MONEY_MATH_CONTEXT), currencyCode);
    }

    public Money multiply(double multiplier) {
        return multiply((new BigDecimal(Math.round(multiplier * ROUND_CORRECTION)))
                .scaleByPowerOfTen(-ROUND_CORRECTION_SCALE));
    }

    public Money divide(BigDecimal divisor) {
        return Money.valueOf(value.divide(divisor, MONEY_MATH_CONTEXT), currencyCode);
    }

    public Money divide(double divisor) {
        return divide(BigDecimal.valueOf(divisor));
    }

    public Money divide(long divisor) {
        return divide(BigDecimal.valueOf(divisor));
    }

    public Money abs() {
        return Money.valueOf(value.abs(MONEY_MATH_CONTEXT), currencyCode);
    }

    /**
     * Получить цену с учётом НДС.
     * <p>
     * Значение округляется до сотых долей вверх.
     * Если НДС 13%, и цена 100 руб, то при вызове этого метода вернётся 113 руб.
     */
    public Money addNds(Percent nds) {
        return add(nds).roundToCentDown();
    }

    /**
     * Получить цену с вычетом НДС.
     * <p>
     * Значение округляется до сотых долей вниз.
     * Если НДС 13%, и цена 113 руб, то при вызове этого метода вернётся 100 руб.
     */
    public Money subtractNds(Percent nds) {
        // Вычитание НДС чуть хитрее вычитания процентов, так как при добавлении/вычитании НДС должны получить
        // исходную сумму
        return this.divide(BigDecimal.ONE.add(nds.asRatio())).roundToCentDown();
    }

    /**
     * Возвращает значение, округлённое вверх до сотых долей валюты (центов, копеек, ...)
     */
    public Money roundToCentUp() {
        return Money.valueOf(value.setScale(MONEY_CENT_SCALE, RoundingMode.UP), currencyCode);
    }

    /**
     * Возвращает значение, округлённое вниз до сотых долей валюты (центов, копеек, ...)
     */
    public Money roundToCentDown() {
        return Money.valueOf(value.setScale(MONEY_CENT_SCALE, RoundingMode.DOWN), currencyCode);
    }

    /**
     * Возвращает {@link Money}, приведённую к границам [min; max] из настройки соответствующей валюты.
     * <p>
     * Псевдокод:
     * <pre>
     * return if (value > maxPrice) {
     *   maxPrice
     * } else if (value < minPrice) {
     *   minPrice
     * } else {
     *   value
     * }</pre>
     *
     * @see Currency#getMinPrice()
     * @see Currency#getMaxPrice()
     */
    public Money adjustToCurrencyRange() {
        Currency currency = currencyCode.getCurrency();
        BigDecimal minPrice = currency.getMinPrice();
        BigDecimal maxPrice = currency.getMaxPrice();
        if (value.compareTo(minPrice) < 0) {
            return Money.valueOf(minPrice, currencyCode);
        } else if (value.compareTo(maxPrice) > 0) {
            return Money.valueOf(maxPrice, currencyCode);
        } else {
            return this;
        }
    }

    public Money adjustToCPMRange() {
        Currency currency = currencyCode.getCurrency();
        BigDecimal minPrice = currency.getMinCpmPrice();
        BigDecimal maxPrice = currency.getMaxPrice();
        if (value.compareTo(minPrice) < 0) {
            return Money.valueOf(minPrice, currencyCode);
        } else if (value.compareTo(maxPrice) > 0) {
            return Money.valueOf(maxPrice, currencyCode);
        } else {
            return this;
        }
    }

    /**
     * @return {@code true} если значение {@link Money} попадает в диапазон допустимых значений ставки для валюты.
     * {@code false} в противном случае.
     * @see Currency#getMinPrice()
     * @see Currency#getMaxPrice()
     */
    public boolean isInCurrencyRange() {
        Currency currency = currencyCode.getCurrency();
        BigDecimal minPrice = currency.getMinPrice();
        BigDecimal maxPrice = currency.getMaxPrice();
        return value.compareTo(minPrice) >= 0 && value.compareTo(maxPrice) <= 0;
    }

    /**
     * Возвращает {@link Money}, приведённую к границам [min; max] из настройки соответствующей валюты.
     * За верхнюю границу берется значение максимальной отображаемой ставки из торгов
     * <p>
     * Псевдокод:
     * <pre>
     * return if (value > maxPrice) {
     *   maxPrice
     * } else if (value < minPrice) {
     *   minPrice
     * } else {
     *   value
     * }</pre>
     *
     * @see Currency#getMinPrice()
     * @see Currency#getMaxShowBid()
     */
    public Money adjustToAuctionBidRange() {
        Currency currency = currencyCode.getCurrency();
        BigDecimal minPrice = currency.getMinPrice();
        BigDecimal maxPrice = currency.getMaxShowBid();
        if (value.compareTo(minPrice) < 0) {
            return Money.valueOf(minPrice, currencyCode);
        } else if (value.compareTo(maxPrice) > 0) {
            return Money.valueOf(maxPrice, currencyCode);
        } else {
            return this;
        }
    }

    /**
     * Округляет сумму вниз до шага аукциона для ее валюты
     *
     * @return округленная сумма
     */
    public Money roundToAuctionStepDown() {
        return roundToAuctionStep(RoundingMode.DOWN);
    }

    /**
     * Округляет сумму вниз до шага аукциона для ее валюты
     *
     * @return округленная сумма
     */
    public Money roundToAuctionStepUp() {
        return roundToAuctionStep(RoundingMode.UP);
    }

    /**
     * Округляет сумму до шага аукциона для ее валюты
     *
     * @param mode режим округления
     * @return округленная сумма
     */
    public Money roundToAuctionStep(RoundingMode mode) {
        BigDecimal auctionStep = currencyCode.getCurrency().getAuctionStep();
        return Money.valueOf(
                this.bigDecimalValue()
                        .divide(auctionStep, 0, mode)
                        .multiply(auctionStep),
                currencyCode);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        Money money = (Money) o;
        return value.compareTo(money.value) == 0 &&
                currencyCode == money.currencyCode;
    }

    @Override
    public int hashCode() {
        return Objects.hash(value, currencyCode);
    }

    @Override
    public String toString() {
        return "Money{" +
                "value=" + value +
                ", currencyCode=" + currencyCode +
                '}';
    }
}
