package ru.yandex.direct.currency;

import java.math.BigDecimal;
import java.math.MathContext;
import java.math.RoundingMode;

import com.google.common.math.LongMath;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.currency.Money.MICRO_MULTIPLIER_SCALE;

/**
 * Утилитные методы для работы с несколькими экземплярами {@link Money}. Например, поиск минимальной/максимальной цены.
 */
public class MoneyUtils {
    public static final int INTERNAL_MONEY_SCALE = 4;
    public static final int ROUND_CORRECTION_SCALE = 8;
    public static final long ROUND_CORRECTION = LongMath.pow(10, ROUND_CORRECTION_SCALE);
    public static final BigDecimal NEW_MAX_AUTOPAY_CARD = new BigDecimal("299999.99");
    /**
     * @return аргумент, меньший по величине. При равенстве возвращается первый аргумент
     * @throws IllegalArgumentException если валюты у переданных аргументов отличаются
     */
    public static Money min(Money left, Money right) throws IllegalArgumentException {
        checkMoneyCurrencyCodesAreSame(left, right);

        return left.lessThanOrEqual(right)
                ? left
                : right;
    }

    /**
     * @return аргумент, больший по величине. При равенстве возвращается первый аргумент
     * @throws IllegalArgumentException если валюты у переданных аргументов отличаются
     */
    public static Money max(Money left, Money right) throws IllegalArgumentException {
        checkMoneyCurrencyCodesAreSame(left, right);

        return right.lessThanOrEqual(left)
                ? left
                : right;
    }

    /**
     * @throws IllegalArgumentException если валюты у переданных аргументов отличаются
     */
    static void checkMoneyCurrencyCodesAreSame(Money left, Money right) {
        checkArgument(left.getCurrencyCode() == right.getCurrencyCode(), "Can't compare money of different currencies");
    }

    public static double roundToAuctionStepUpDouble(double value, double auctionStep) {
        return Math.round(Math.ceil(value / auctionStep) * auctionStep * ROUND_CORRECTION) / ((double) ROUND_CORRECTION);
    }

    public static BigDecimal roundToAuctionStep(BigDecimal val, Currency cur, RoundingMode mode) {
        BigDecimal auctionStep = cur.getAuctionStep();

        return val
                .divide(auctionStep, 0, mode)
                .multiply(auctionStep);
    }

    public static long roundMicrosToAuctionStep(long val, Currency cur, RoundingMode mode) {
        long auctionStep = cur.getAuctionStep().scaleByPowerOfTen(MICRO_MULTIPLIER_SCALE).longValue();
        return LongMath.divide(val, auctionStep, mode) * auctionStep;
    }

    /**
     * Вычитает процент из заданной суммы
     *
     * Включаем в DIRECT-149354
     * После включения выпиливаем в DIRECT-149770
     *
     * @param sum     - сумма, из которой нужно вычесть проценты
     * @param percent - вычитаемый процент
     */
    public static BigDecimal subtractPercentPrecisely(BigDecimal sum, Percent percent) {
        return sum.divide(BigDecimal.ONE.add(percent.asRatio()), MathContext.DECIMAL64)
                .setScale(INTERNAL_MONEY_SCALE, RoundingMode.DOWN);
    }

    /**
     * При включенном флаге для рублей подменяем лимит на автоплатеж картой.
     *
     *
     * @param curr
     * @param useNewAutopayCardLimit - флаг, использовать ли новый лимит NEW_MAX_AUTOPAY_CARD
     * @return
     */
    public static BigDecimal getMaxAutopayCard(Currency curr, boolean useNewAutopayCardLimit) {
        if (useNewAutopayCardLimit && CurrencyCode.RUB == curr.getCode()) {
            return NEW_MAX_AUTOPAY_CARD;
        }
        return curr.getMaxAutopayCard();
    }
}
