package ru.yandex.direct.currency;

import java.io.IOException;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Objects;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.JsonSerializer;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;

/**
 * Тип для процентов
 */
@JsonSerialize(using = Percent.Serializer.class)
@JsonDeserialize(using = Percent.Deserializer.class)
public class Percent {
    private static final BigDecimal ONE_HUNDRED = BigDecimal.valueOf(100L);

    private final BigDecimal ratioValue;

    private Percent(BigDecimal ratioValue) {
        this.ratioValue = ratioValue;
    }

    /**
     * @param ratioValue значение в размерности процентов [0, 100]
     */
    public static Percent fromPercent(BigDecimal ratioValue) {
        return new Percent(ratioValue.divide(ONE_HUNDRED, 4, RoundingMode.UP));
    }

    /**
     * @param ratioValue значение в размерности долей [0.0, 1.0]
     */
    public static Percent fromRatio(BigDecimal ratioValue) {
        return new Percent(ratioValue);
    }

    /**
     * @return значение как проценты.
     * единица &ndash; {@code 100}
     */
    public BigDecimal asPercent() {
        return ratioValue.multiply(ONE_HUNDRED);
    }

    /**
     * @return значение как долю от единицы.
     * единица &ndash; {@code 1}
     */
    public BigDecimal asRatio() {
        return ratioValue;
    }

    @Override
    public String toString() {
        return "Percent{" +
                "ratioValue=" + ratioValue +
                '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        Percent percent = (Percent) o;
        // Сравниваем через compareTo, потому что equals на BigDecimal смотрит не только на численное значение, но и на точность
        return Objects.compare(ratioValue, percent.ratioValue, BigDecimal::compareTo) == 0;
    }

    @Override
    public int hashCode() {
        // hashCode вычисляем по doubleValue(), чтобы обеспечить выполнение контракта equals/hashCode
        return Double.hashCode(ratioValue.doubleValue());
    }

    /**
     * Миниатюрный JSON-сериализатор для {@link Percent}.
     * <p>
     * Значение процентов округляется до 2 знаков после запятой.
     * Стратегия округления &ndash; {@link RoundingMode#HALF_EVEN}
     */
    public static class Serializer extends JsonSerializer<Percent> {

        @Override
        public void serialize(Percent value, JsonGenerator gen, SerializerProvider serializers) throws IOException {
            gen.writeNumber(value.asPercent().setScale(2, RoundingMode.HALF_EVEN));
        }

    }

    /**
     * Миниатюрный JSON-десериализатор для {@link Percent}.
     */
    public static class Deserializer extends JsonDeserializer<Percent> {

        @Override
        public Percent deserialize(JsonParser p, DeserializationContext ctxt) throws IOException {
            return Percent.fromPercent(new BigDecimal(p.getText()));
        }
    }
}
