package ru.yandex.direct.currency;

import java.time.LocalDate;
import java.util.Collections;
import java.util.List;
import java.util.ListIterator;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;


/**
 * Утилитные методы для работы с курсами {@link FixedRates}. Например, получение текущего курса для валюты.
 */
public class RateUtils {

    private static final ImmutableMap<CurrencyCode, List<Rate>> FIXED_RATES =
            new ImmutableMap.Builder<CurrencyCode, List<Rate>>()
                    .put(CurrencyCode.YND_FIXED, Collections.singletonList(FixedRates.ONE))
                    .put(CurrencyCode.BYN, ImmutableList.sortedCopyOf(FixedRates.BYN_HISTORY))
                    .put(CurrencyCode.CHF, ImmutableList.sortedCopyOf(FixedRates.CHF_HISTORY))
                    .put(CurrencyCode.EUR, ImmutableList.sortedCopyOf(FixedRates.EUR_HISTORY))
                    .put(CurrencyCode.KZT, ImmutableList.sortedCopyOf(FixedRates.KZT_HISTORY))
                    .put(CurrencyCode.RUB, ImmutableList.sortedCopyOf(FixedRates.RUB_HISTORY))
                    .put(CurrencyCode.TRY, ImmutableList.sortedCopyOf(FixedRates.TRY_HISTORY))
                    .put(CurrencyCode.UAH, ImmutableList.sortedCopyOf(FixedRates.UAH_HISTORY))
                    .put(CurrencyCode.USD, ImmutableList.sortedCopyOf(FixedRates.USD_HISTORY))
                    .put(CurrencyCode.GBP, ImmutableList.sortedCopyOf(FixedRates.GBP_HISTORY))
                    .build();

    /**
     * @return курс валюты на дату или null, если курс отсутствует
     */
    public static Rate get(CurrencyCode currencyCode, LocalDate date) {
        List<Rate> rates = FIXED_RATES.get(currencyCode);
        ListIterator<Rate> li = rates.listIterator(rates.size());

        while (li.hasPrevious()) {
            Rate rate = li.previous();
            if (!rate.getDate().isAfter(date)) {
                return rate;
            }
        }

        return null;
    }

    /**
     * @return курс валюты на текущую дату или null, если курс отсутствует
     */
    public static Rate get(CurrencyCode currencyCode) {
        return get(currencyCode, LocalDate.now());
    }
}
