package ru.yandex.direct.intapi.client;

import java.math.BigDecimal;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;
import com.google.common.primitives.Ints;
import one.util.streamex.StreamEx;
import org.apache.http.client.utils.URIBuilder;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.RequestBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.asynchttp.FetcherSettings;
import ru.yandex.direct.asynchttp.ParallelFetcher;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.ParsableStringRequest;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.intapi.client.model.handle.BaseStringJsonIntApiHandle;
import ru.yandex.direct.intapi.client.model.handle.BsFrontIntapiHandle;
import ru.yandex.direct.intapi.client.model.handle.CampaignStatisticsIntApiHandle;
import ru.yandex.direct.intapi.client.model.handle.CampaignsCopyIntApiHandle;
import ru.yandex.direct.intapi.client.model.handle.IntApiHandle;
import ru.yandex.direct.intapi.client.model.handle.PayForAllIntApiHandle;
import ru.yandex.direct.intapi.client.model.handle.PrepareAndValidatePayCampIntApiHandle;
import ru.yandex.direct.intapi.client.model.request.CalculateCampaignStatusModerateRequest;
import ru.yandex.direct.intapi.client.model.request.CampaignUnarcRequest;
import ru.yandex.direct.intapi.client.model.request.CampaignsCopyRequest;
import ru.yandex.direct.intapi.client.model.request.IntApiRequest;
import ru.yandex.direct.intapi.client.model.request.NotificationRequest;
import ru.yandex.direct.intapi.client.model.request.PayForAllRequest;
import ru.yandex.direct.intapi.client.model.request.PrepareAndValidatePayCampRequest;
import ru.yandex.direct.intapi.client.model.request.bsfront.BsFrontRequest;
import ru.yandex.direct.intapi.client.model.request.bsfront.Creative;
import ru.yandex.direct.intapi.client.model.request.bsfront.Params;
import ru.yandex.direct.intapi.client.model.request.statistics.CampaignStatisticsRequest;
import ru.yandex.direct.intapi.client.model.response.CampStatusModerate;
import ru.yandex.direct.intapi.client.model.response.CampaignsCopyResponse;
import ru.yandex.direct.intapi.client.model.response.PayForAllResponse;
import ru.yandex.direct.intapi.client.model.response.PrepareAndValidatePayCampResponse;
import ru.yandex.direct.intapi.client.model.response.bsfront.BsFrontResponse;
import ru.yandex.direct.intapi.client.model.response.statistics.CampaignStatisticsResponse;
import ru.yandex.direct.solomon.SolomonUtils;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceChild;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.utils.JsonUtils;

import static io.netty.handler.codec.http.HttpHeaderNames.CONTENT_TYPE;
import static org.asynchttpclient.util.HttpConstants.Methods.POST;
import static ru.yandex.direct.tracing.util.TraceUtil.X_YANDEX_TRACE;
import static ru.yandex.direct.tracing.util.TraceUtil.traceToHeader;

/**
 * Клиент для внутреннего API Директа, предназначен для вызова тех кусков кода в перловом Директе, которые
 * нельзя пока перенести в Java-Директ по каким угодно причинам
 */
public class IntApiClient {
    private static final String PATH_DELIMETER = "/";
    static final String SERVICE_TICKET_HEADER = "X-Ya-Service-Ticket";
    static final String CAMPAIGN_UNARC_SUCCESS_RESULT = "1";
    static final IntApiHandle<String> CAMPAIGN_UNARC_HANDLE = new BaseStringJsonIntApiHandle("CampaignUnarc");
    static final IntApiHandle<String> NOTIFICATION_HANDLE = new BaseStringJsonIntApiHandle("Notification");
    static final IntApiHandle<String> CALCULATE_CAMPAIGN_STATUS = new BaseStringJsonIntApiHandle(
            "CalculateCampaignStatusModerate");
    static final IntApiHandle<String> CALCULATE_CAMPAIGN_STATUS_READONLY = new BaseStringJsonIntApiHandle(
            "CalculateCampaignStatusModerateReadOnly");

    // Количество кампаний которое можно обработать за раз, рассчитано примерно по трейс логам -
    // сколько мы можем обработать кампаний за 15 секунд
    static final int CALCULATE_CAMPAIGN_STATUS_CHUNK_SIZE = 50;

    private static final IntApiHandle<PayForAllResponse> PAY_FOR_ALL_HANDLE =
            new PayForAllIntApiHandle("PayForAll");
    private static final IntApiHandle<PrepareAndValidatePayCampResponse> VALIDATE_PAY_CAMP_HANDLE =
            new PrepareAndValidatePayCampIntApiHandle("PrepareAndValidatePayCamp");
    private static final IntApiHandle<CampaignStatisticsResponse> CAMPAIGN_STATISTICS_HANDLE =
            new CampaignStatisticsIntApiHandle("CampaignStatistics");
    static final IntApiHandle<BsFrontResponse> BS_FRONT_HANDLE = new BsFrontIntapiHandle();
    private static final IntApiHandle<CampaignsCopyResponse> CAMPAINGS_COPY_HANDLE =
            new CampaignsCopyIntApiHandle("CampaignsCopy");

    private static final Logger logger = LoggerFactory.getLogger(IntApiClient.class);

    private final IntApiClientConfiguration config;
    private final ParallelFetcherFactory parallelFetcherFactory;
    private final Supplier<String> tvmServiceTicketProvider;

    public IntApiClient(IntApiClientConfiguration config, AsyncHttpClient asyncHttpClient,
                        Supplier<String> tvmServiceTicketProvider) {
        this.config = config;
        this.tvmServiceTicketProvider = tvmServiceTicketProvider;
        FetcherSettings settings = new FetcherSettings()
                .withRequestTimeout(config.getReadTimeout());
        this.parallelFetcherFactory = new ParallelFetcherFactory(asyncHttpClient, settings);
    }

    public IntApiClient(IntApiClientConfiguration config, ParallelFetcherFactory parallelFetcherFactory,
                        Supplier<String> tvmServiceTicketProvider) {
        this.config = config;
        this.parallelFetcherFactory = parallelFetcherFactory;
        this.tvmServiceTicketProvider = tvmServiceTicketProvider;
    }

    private String buildUrl(String path) {
        String realPath = path.startsWith(PATH_DELIMETER) ? path : PATH_DELIMETER + path;
        try {
            return new URIBuilder(config.getUri()).setPath(realPath).build().toASCIIString();
        } catch (URISyntaxException e) {
            throw new IntApiClientException(e);
        }
    }

    /**
     * Сделать запрос к интапи-ручке с заданным таймаутом и вернуть ее ответ в виде объекта ответа ручки,
     * заданного в ее спеке.
     */

    @Nonnull
    <T> T doRequest(IntApiHandle<T> handle, IntApiRequest request, Duration timeout) {
        return doRequest(handle, List.of(request), timeout).get(0);
    }

    @Nonnull
    <T> List<T> doRequest(IntApiHandle<T> handle, List<? extends IntApiRequest> requests, Duration timeout) {
        Map<Long, Result<String>> resultMap;

        try (
                TraceProfile profile = Trace.current().profile("intapi_client", handle.getPath());
                TraceChild traceChild = Trace.current().child("direct.intapi", handle.getPath());
                ParallelFetcher<String> fetcher = parallelFetcherFactory.getParallelFetcherWithMetricRegistry(
                        SolomonUtils.getParallelFetcherMetricRegistry(profile.getFunc()))) {

            resultMap = fetcher.execute(buildParsableRequestsList(handle, requests, timeout, traceChild));

            for (var result : resultMap.values()) {
                if (result.getErrors() != null && !result.getErrors().isEmpty()) {
                    logger.error("Got errors: {}", result.getErrors());
                    RuntimeException ex = new IntApiClientException("Got error on response for IntApi request");
                    result.getErrors().forEach(ex::addSuppressed);
                    throw ex;
                }
            }

        } catch (InterruptedException ex) {
            logger.error("Request were unexpectedly interrupted");
            Thread.currentThread().interrupt();
            throw new IntApiClientException(ex);
        }

        return deserializeResults(handle, resultMap.values());
    }

    <T> List<T> deserializeResults(IntApiHandle<T> handle, Collection<Result<String>> results) {
        List<T> deserializedResults = new ArrayList<>();

        for (var result : results) {
            T response = handle.deserializeResponse(result.getSuccess());

            if (response == null) {
                throw new IntApiClientException("Deserialization result is null");
            }

            deserializedResults.add(response);
        }

        return deserializedResults;
    }

    <T> List<ParsableStringRequest> buildParsableRequestsList(IntApiHandle<T> handle,
                                                              List<? extends IntApiRequest> requests,
                                                              Duration timeout, TraceChild traceChild) {

        List<ParsableStringRequest> results = new ArrayList<>();
        long reqId = 1;

        for (var r : requests) {
            results.add(buildOneRequest(reqId++, handle, r, timeout, traceChild));
        }

        return results;
    }

    <T> ParsableStringRequest buildOneRequest(long requestId, IntApiHandle<T> handle, IntApiRequest request,
                                              Duration timeout, TraceChild traceChild) {
        RequestBuilder builder = new RequestBuilder(POST)
                .setRequestTimeout(Ints.checkedCast(timeout.toMillis()))
                .setUrl(buildUrl(handle.getPath()))
                .setCharset(StandardCharsets.UTF_8)
                .setBody(handle.serializeRequest(request))
                .addHeader(CONTENT_TYPE, handle.getContentType())
                .addHeader(X_YANDEX_TRACE, traceToHeader(traceChild));

        if (tvmServiceTicketProvider.get() != null) {
            builder.addHeader(SERVICE_TICKET_HEADER, tvmServiceTicketProvider.get());
        }

        logger.trace("Request body: {}", request);

        return new ParsableStringRequest(requestId, builder.build());
    }

    /**
     * Сделать запрос к интапи-ручке Notification со стандартным таймаутом.
     */
    public void addNotification(String notificationType, Map<String, Object> data, Map<String, Object> options) {
        addNotification(notificationType, data, options, config.getReadTimeout());
    }

    /**
     * Сделать запрос к интапи-ручке Notification с заданным таймаутом.
     */
    public void addNotification(String notificationType, Map<String, Object> data, Map<String, Object> options,
                                Duration timeout) {
        NotificationRequest request = new NotificationRequest(notificationType, data, options);

        String result = doRequest(NOTIFICATION_HANDLE, request, timeout);

        if (!result.isEmpty()) {
            logger.error("Got unexpected response: \"{}\"", result);
            throw new IntApiClientException("Got unexpected response for Notification request");
        }
    }

    /**
     * Сделать запрос к интапи-ручке CampaignUnarc со стандартным таймаутом.
     */
    public void unarcCampaign(Long uid, Long cid, Boolean force) {
        unarcCampaign(uid, cid, force, config.getReadTimeout());
    }

    /**
     * Сделать запрос к интапи-ручке CalculateCampaignStatusModerate со стандартным таймаутом.
     */
    public void calculateCampaignStatusModerate(List<Long> cids) {
        calculateCampaignStatusModerate(cids, Duration.ofMinutes(5), CALCULATE_CAMPAIGN_STATUS_CHUNK_SIZE);
    }

    /**
     * Сделать запрос к интапи-ручке CalculateCampaignStatusModerateReadOnly.
     */
    public Map<Long, CampStatusModerate> calculateCampaignStatusModerateReadOnly(List<Long> cids) {
        return calculateCampaignStatusModerateReadOnly(cids, Duration.ofMinutes(5),
                CALCULATE_CAMPAIGN_STATUS_CHUNK_SIZE);
    }

    /**
     * Сделать запрос к интапи-ручке BsFront.change_notify
     */
    public void changeNotifyCreatives(String login, long uid, List<Integer> creativeIds) {
        BsFrontRequest request = new BsFrontRequest().withMethod("change_notify").withParams(
                new Params().withClientLogin(login).withOperatorUid(uid).withCreatives(
                        StreamEx.of(creativeIds)
                                .map(id -> new Creative().withId(id))
                                .toList()
                )
        );

        BsFrontResponse result = doRequest(BS_FRONT_HANDLE, request, config.getReadTimeout());
        if (result.getResult() == null || result.getResult().stream().anyMatch(r -> r.getResult() != 1)) {
            logger.error("Got unexpected response: \"{}\"", result);
            throw new IntApiClientException("Got some errors from BsFront.change_notify " + result.toString());
        }
    }

    /**
     * Сделать запрос к интапи-ручке CampaignUnarc с заданным таймаутом.
     */
    public void unarcCampaign(Long uid, Long cid, Boolean force, Duration timeout) {
        CampaignUnarcRequest request = new CampaignUnarcRequest(uid, cid, force);

        String result = doRequest(CAMPAIGN_UNARC_HANDLE, request, timeout);
        if (!result.equals(CAMPAIGN_UNARC_SUCCESS_RESULT)) {
            logger.error("Got unexpected response: \"{}\"", result);
            throw new IntApiClientException("Got unexpected response for CampaignUnarc request");
        }
    }

    /**
     * Запрос к ручке копирования кампании CampaignsCopy.
     */
    public CampaignsCopyResponse copyCampaigns(Long operatorUid, Long clientIdFrom, Long clientIdTo,
                                               List<Long> campaignIds, Locale locale, boolean sync) {
        CampaignsCopyRequest request = new CampaignsCopyRequest(operatorUid, clientIdFrom, clientIdTo, campaignIds,
                locale.toString().toLowerCase(), sync);
        return doRequest(CAMPAINGS_COPY_HANDLE, request,
                sync ? config.getCopyCampaignsReadTimeout() : config.getReadTimeout());
    }

    /**
     * Сделать запрос к интапи-ручке CalculateCampaignStatusModerate с заданным таймаутом.
     */
    public void calculateCampaignStatusModerate(List<Long> cids, Duration timeout, int chunkSize) {
        List<CalculateCampaignStatusModerateRequest> requests = Lists.partition(cids, chunkSize)
                .stream().map(CalculateCampaignStatusModerateRequest::new).collect(Collectors.toList());

        List<String> results = doRequest(CALCULATE_CAMPAIGN_STATUS, requests, timeout);

        for (String result : results) {
            if (!result.equals(CAMPAIGN_UNARC_SUCCESS_RESULT)) {
                logger.error("Got unexpected response: \"{}\"", result);
                throw new IntApiClientException("Got unexpected response for CalculateCampaignStatusModerate request");
            }
        }
    }

    private Map<Long, CampStatusModerate> calculateCampaignStatusModerateReadOnly(List<Long> cids,
                                                                                  Duration timeout,
                                                                                  int chunkSize) {
        List<CalculateCampaignStatusModerateRequest> requests = Lists.partition(cids, chunkSize)
                .stream().map(CalculateCampaignStatusModerateRequest::new).collect(Collectors.toList());

        List<String> results = doRequest(CALCULATE_CAMPAIGN_STATUS_READONLY, requests, timeout);

        Map<Long, CampStatusModerate> resultMap = new HashMap<>();
        for (String result : results) {
            Map<Long, CampStatusModerate> map = JsonUtils.fromJson(result,
                    new TypeReference<>() {
                    });
            resultMap.putAll(map);
        }
        return resultMap;
    }

    /**
     * Валидация оплаты кампании перловым кодом, MoneyTransfer::prepare_and_validate_pay_camp
     * https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/MoneyTransfer.pm#L716
     */
    public boolean validatePayCamp(Long uid, Long cid, Long operatorUid, BigDecimal sum) {
        PrepareAndValidatePayCampRequest request = new PrepareAndValidatePayCampRequest()
                .withUid(uid)
                .withCid(cid)
                .withOperatorUid(operatorUid)
                .withSum(sum);
        PrepareAndValidatePayCampResponse response = doRequest(VALIDATE_PAY_CAMP_HANDLE, request,
                config.getReadTimeout());

        logger.info("Got PrepareAndValidatePayCamp response: error = {}, error_code = {}",
                response.getError(), response.getErrorCode());

        return (response.getErrorCode() == null) && (response.getError() == null || response.getError().equals("null"));
    }

    /**
     * Выставление счета, перл, DoCmd::cmd_payforall
     */
    public PayForAllResponse payForAll(
        String role,
        Long uid,
        String login,
        Long cid,
        Long operatorUid,
        BigDecimal sum,
        String requestUrl,
        Boolean withNds,
        Locale locale
    ) {
        PayForAllRequest request = new PayForAllRequest()
                .withUid(uid)
                .withCid(cid)
                .withOperatorUid(operatorUid)
                .withSum(sum)
                .withLogin(login)
                .withRequestUrl(requestUrl)
                .withRole(role)
                .withNds(withNds)
                .withLanguage(locale.getLanguage());
        PayForAllResponse response = doRequest(PAY_FOR_ALL_HANDLE, request,
                config.getReadTimeout());

        logger.info("Got PayForAll response: error = {}, error_code = {}, billing_url = {}",
                response.getError(), response.getErrorCode(), response.getUrl());

        return response;
    }


    public CampaignStatisticsResponse getCampaignStatistics(CampaignStatisticsRequest request) {
        CampaignStatisticsResponse response = doRequest(CAMPAIGN_STATISTICS_HANDLE, request,
                config.getReadTimeout());

        logger.info("Got CampaignStatistics response: error = {}, error_code = {}",
                response.getError(), response.getErrorCode());

        return response;
    }
}
