package ru.yandex.direct.intapi.client.model.request.statistics;

import java.time.LocalDate;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonFormat;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;

import ru.yandex.direct.intapi.client.model.request.statistics.option.ReportOptionColumn;
import ru.yandex.direct.intapi.client.model.request.statistics.option.ReportOptionConditionOperator;
import ru.yandex.direct.intapi.client.model.request.statistics.option.ReportOptionFilterColumn;
import ru.yandex.direct.intapi.client.model.request.statistics.option.ReportOptionGroupBy;
import ru.yandex.direct.intapi.client.model.request.statistics.option.ReportOptionGroupByDate;
import ru.yandex.direct.intapi.client.model.request.statistics.option.ReportOptionOrder;
import ru.yandex.direct.intapi.client.model.request.statistics.option.ReportOptionOrderByField;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.intapi.client.model.request.statistics.option.ReportOptionFilterColumn.GOAL_IDS;

@JsonInclude(JsonInclude.Include.NON_NULL)
public class ReportOptions {

    @JsonProperty("cid")
    private Long campaignId;

    @JsonProperty("filters")
    private Map<ReportOptionFilterColumn, Map<ReportOptionConditionOperator, ?>> filters;

    @JsonProperty("date_from")
    @JsonFormat(pattern = "yyyy-MM-dd")
    private LocalDate dateFrom;

    @JsonProperty("date_to")
    @JsonFormat(pattern = "yyyy-MM-dd")
    private LocalDate dateTo;

    @JsonProperty("date_from_b")
    @JsonFormat(pattern = "yyyy-MM-dd")
    private LocalDate dateFromB;

    @JsonProperty("date_to_b")
    @JsonFormat(pattern = "yyyy-MM-dd")
    private LocalDate dateToB;

    @JsonProperty("group_by")
    private Set<ReportOptionGroupBy> groupBy;

    @JsonProperty("group_by_date")
    private ReportOptionGroupByDate groupByDate;

    @JsonProperty("columns")
    private Set<ReportOptionColumn> columns;

    @JsonProperty("compare_periods")
    private Integer comparePeriods;

    @JsonProperty("stat_type")
    private ReportType reportType;

    @JsonProperty("limit")
    private Integer limit;

    @JsonProperty("offset")
    private Integer offset;

    @JsonProperty("order_by")
    private ReportOptionOrderByField orderByField;

    @JsonProperty("order_dir")
    private ReportOptionOrder order;

    public ReportOptions withCampaignId(Long campaignId) {
        this.campaignId = campaignId;
        return this;
    }

    public ReportOptions withFilters(Map<ReportOptionFilterColumn, Map<ReportOptionConditionOperator, ?>> filters) {
        this.filters = filters;
        return this;
    }

    public ReportOptions withDateFrom(LocalDate dateFrom) {
        this.dateFrom = dateFrom;
        return this;
    }

    public ReportOptions withDateTo(LocalDate dateTo) {
        this.dateTo = dateTo;
        return this;
    }

    public ReportOptions withDateFromB(LocalDate dateFromB) {
        this.dateFromB = dateFromB;
        return this;
    }

    public ReportOptions withDateToB(LocalDate dateToB) {
        this.dateToB = dateToB;
        return this;
    }

    public ReportOptions withGroupBy(Set<ReportOptionGroupBy> groupBy) {
        this.groupBy = groupBy;
        return this;
    }

    public ReportOptions withGroupByDate(ReportOptionGroupByDate groupByDate) {
        this.groupByDate = groupByDate;
        return this;
    }

    public ReportOptions withColumns(Set<ReportOptionColumn> columns) {
        this.columns = columns;
        return this;
    }

    public ReportOptions comparePeriods() {
        this.comparePeriods = 1;
        return this;
    }

    public ReportOptions withReportType(ReportType reportType) {
        this.reportType = reportType;
        return this;
    }

    public ReportOptions withLimit(Integer limit) {
        this.limit = limit;
        return this;
    }

    public ReportOptions withOffset(Integer offset) {
        this.offset = offset;
        return this;
    }

    public ReportOptions withOrderByField(ReportOptionOrderByField orderByField) {
        this.orderByField = orderByField;
        return this;
    }

    public ReportOptions withOrder(ReportOptionOrder order) {
        this.order = order;
        return this;
    }

    public Set<ReportOptionGroupBy> getGroupBy() {
        return groupBy;
    }

    public ReportOptionGroupByDate getGroupByDate() {
        return groupByDate;
    }

    public Long getCampaignId() {
        return campaignId;
    }

    public LocalDate getDateFrom() {
        return dateFrom;
    }

    public LocalDate getDateTo() {
        return dateTo;
    }

    @JsonIgnore
    public List<Long> getGoalIds() {
        var goalIdsFilter = filters.get(GOAL_IDS);

        if (goalIdsFilter == null) {
            return emptyList();
        }

        return (List<Long>) goalIdsFilter.get(ReportOptionConditionOperator.EQ);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ReportOptions that = (ReportOptions) o;
        return Objects.equals(campaignId, that.campaignId) &&
                Objects.equals(filters, that.filters) &&
                Objects.equals(dateFrom, that.dateFrom) &&
                Objects.equals(dateTo, that.dateTo) &&
                Objects.equals(dateFromB, that.dateFromB) &&
                Objects.equals(dateToB, that.dateToB) &&
                Objects.equals(groupBy, that.groupBy) &&
                groupByDate == that.groupByDate &&
                Objects.equals(columns, that.columns) &&
                Objects.equals(comparePeriods, that.comparePeriods) &&
                reportType == that.reportType &&
                Objects.equals(limit, that.limit) &&
                Objects.equals(offset, that.offset) &&
                orderByField == that.orderByField &&
                order == that.order;
    }

    @Override
    public int hashCode() {
        return Objects.hash(campaignId, filters, dateFrom, dateTo, dateFromB, dateToB, groupBy, groupByDate, columns,
                comparePeriods, reportType, limit, offset, orderByField, order);
    }
}
