package ru.yandex.direct.display.landing.client;

import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import org.asynchttpclient.RequestBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.asynchttp.ParallelFetcher;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.ParsableRequest;
import ru.yandex.direct.asynchttp.ParsableStringRequest;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.display.landing.client.submissions.GetSubmissionsRequest;
import ru.yandex.direct.display.landing.client.submissions.GetSubmissionsResponse;
import ru.yandex.direct.solomon.SolomonUtils;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceChild;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.utils.InterruptedRuntimeException;

import static io.netty.handler.codec.http.HttpHeaderNames.AUTHORIZATION;
import static ru.yandex.direct.tracing.util.TraceUtil.X_YANDEX_TRACE;
import static ru.yandex.direct.tracing.util.TraceUtil.traceToHeader;

/**
 * Клиент для сервиса Display Landing Backend (BLanding)
 */
@ParametersAreNonnullByDefault
public class DisplayLandingClient {
    private static final Logger logger = LoggerFactory.getLogger(DisplayLandingClient.class);

    private final DisplayLandingClientSettings settings;
    private final ParallelFetcherFactory fetcherFactory;
    private final Supplier<Boolean> useNewSubmissionsUrlSupplier;

    public DisplayLandingClient(DisplayLandingClientSettings settings, ParallelFetcherFactory fetcherFactory,
                                Supplier<Boolean> useNewSubmissionsUrlSupplier) {
        this.settings = settings;
        this.fetcherFactory = fetcherFactory;
        this.useNewSubmissionsUrlSupplier = useNewSubmissionsUrlSupplier;
    }

    /**
     * Возвращает заявки, оставленные на лендингах клиента.
     *
     * @param request параметры запроса - id клиента, список лендингов
     * @return ответ, содержащий список заявок или информацию о возникшей ошибке
     * @throws DisplayLandingClientException в случае возникновения ошибок при обращении к Дисплею
     * @throws InterruptedRuntimeException   в случае прерывания операции системой
     */
    public GetSubmissionsResponse getSubmissions(GetSubmissionsRequest request) {
        try (TraceProfile profile = Trace.current().profile("displayLanding:getSubmissions");
             TraceChild child = Trace.current().child("displayLanding", "getSubmissions");
             ParallelFetcher<String> parallelFetcher = fetcherFactory.getParallelFetcherWithMetricRegistry(
                     SolomonUtils.getParallelFetcherMetricRegistry(profile.getFunc()))) {

            logger.info("Request retries configured: {}", parallelFetcher.getConfiguredRequestRetries());

            String url = useNewSubmissionsUrlSupplier.get() ? settings.getSubmissionsGetUrlNew() :
                    settings.getSubmissionsGetUrl();
            RequestBuilder builder = new RequestBuilder("POST")
                    .setUrl(url)
                    .addHeader(AUTHORIZATION, settings.getAuthToken())
                    .addHeader(X_YANDEX_TRACE, traceToHeader(child));
            request.prepare(builder);

            ParsableRequest<String> asyncRequest = new ParsableStringRequest(builder.build());
            logger.info("Display Landing client request body: {}", asyncRequest);
            Result<String> result = parallelFetcher.execute(asyncRequest);
            if (result.getErrors() != null && !result.getErrors().isEmpty()) {
                logger.info("Got errors: {}", result.getErrors());
                RuntimeException ex = new DisplayLandingClientException(
                        "Got error on response for DisplayCanvas request "
                                + asyncRequest
                                + "errors:" + result.getErrors());
                result.getErrors().forEach(ex::addSuppressed);
                throw ex;
            }

            return GetSubmissionsResponse.deserialize(result.getSuccess());

        } catch (InterruptedException ex) {
            logger.error("Request were unexpectedly interrupted.");
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(ex);
        }
    }
}
