package ru.yandex.direct.display.landing.client.submissions;

import java.time.OffsetDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.databind.annotation.JsonNaming;
import org.asynchttpclient.RequestBuilder;

import static io.netty.handler.codec.http.HttpHeaderNames.CONTENT_TYPE;
import static io.netty.handler.codec.http.HttpHeaderValues.APPLICATION_JSON;
import static java.nio.charset.StandardCharsets.UTF_8;
import static ru.yandex.direct.utils.JsonUtils.toJson;

/**
 * Модель данных запроса к DisplayBlanding ручке submissions, метод get
 */
@ParametersAreNonnullByDefault
public class GetSubmissionsRequest {
    @JsonNaming(value = PropertyNamingStrategy.UpperCamelCaseStrategy.class)
    private static class Request {
        private SelectionCriteria selectionCriteria;
        private final Page page;

        public Request() {
            selectionCriteria = new SelectionCriteria();
            page = new Page();
        }

        public SelectionCriteria getSelectionCriteria() {
            return selectionCriteria;
        }

        public Page getPage() {
            return page;
        }
    }

    @JsonNaming(value = PropertyNamingStrategy.UpperCamelCaseStrategy.class)
    @JsonInclude(value = JsonInclude.Include.NON_NULL)
    public static class SelectionCriteria {
        private List<Long> turboPageIds;
        private Long clientId;
        private OffsetDateTime dateTimeFrom;
        private OffsetDateTime dateTimeTo;

        public List<Long> getTurboPageIds() {
            return turboPageIds;
        }

        public Long getClientId() {
            return clientId;
        }

        public String getDateTimeFrom() {
            if (dateTimeFrom == null) {
                return null;
            }
            return dateTimeFrom.format(DateTimeFormatter.ISO_OFFSET_DATE_TIME);
        }

        public String getDateTimeTo() {
            if (dateTimeTo == null) {
                return null;
            }
            return dateTimeTo.format(DateTimeFormatter.ISO_OFFSET_DATE_TIME);
        }

        public SelectionCriteria withTurboPageIds(List<Long> turboPageIds) {
            this.turboPageIds = turboPageIds;
            return this;
        }

        public SelectionCriteria withClientId(Long clientId) {
            this.clientId = clientId;
            return this;
        }

        public SelectionCriteria withDateTimeFrom(OffsetDateTime dateTimeFrom) {
            this.dateTimeFrom = dateTimeFrom;
            return this;
        }

        public SelectionCriteria withDateTimeTo(OffsetDateTime dateTimeTo) {
            this.dateTimeTo = dateTimeTo;
            return this;
        }
    }

    @JsonNaming(value = PropertyNamingStrategy.UpperCamelCaseStrategy.class)
    private static class Page {
        private int limit;
        private int offset;

        public Page() {
            limit = 10_000;
            offset = 0;
        }

        public void setLimit(int limit) {
            this.limit = limit;
        }

        public void setOffset(int offset) {
            this.offset = offset;
        }

        public int getLimit() {
            return limit;
        }

        public int getOffset() {
            return offset;
        }
    }

    private final Request request;

    public GetSubmissionsRequest() {
        request = new Request();
    }

    /**
     * @param turboPageIds список турбо-страниц (лендингов) клиента
     */
    public GetSubmissionsRequest withTurboPageIds(List<Long> turboPageIds) {
        request.selectionCriteria.turboPageIds = turboPageIds;
        return this;
    }

    /**
     * @param clientId id клиента, данные которого запрашиваются
     */
    public GetSubmissionsRequest withClientId(long clientId) {
        request.selectionCriteria.clientId = clientId;
        return this;
    }

    /**
     * @param dateTimeFrom необязательный параметр, дата-время, с которой выбирать заявки
     */
    public GetSubmissionsRequest withDateTimeFrom(OffsetDateTime dateTimeFrom) {
        request.selectionCriteria.dateTimeFrom = dateTimeFrom;
        return this;
    }

    /**
     * @param dateTimeTo необязательный параметр, дата-время, до которой выбирать заявки
     */
    public GetSubmissionsRequest withDateTimeTo(OffsetDateTime dateTimeTo) {
        request.selectionCriteria.dateTimeTo = dateTimeTo;
        return this;
    }

    /**
     * @param limit необязательный параметр, сколько максимум заявок вернуть
     */
    public GetSubmissionsRequest withLimit(int limit) {
        request.page.limit = limit;
        return this;
    }

    /**
     * @param offset необязательный параметр, с какой заявки, соответствующей условиям, начинать отбор
     */
    public GetSubmissionsRequest withOffset(int offset) {
        request.page.offset = offset;
        return this;
    }


    public GetSubmissionsRequest withSelectionCriteria(SelectionCriteria selectionCriteria) {
        request.selectionCriteria = selectionCriteria;
        return this;
    }


    public void prepare(RequestBuilder builder) {
        builder.setHeader(CONTENT_TYPE, APPLICATION_JSON)
                .setBody(toJson(request))
                .setCharset(UTF_8);
    }
}
