package ru.yandex.direct.dssclient;

import java.math.BigInteger;
import java.util.Arrays;
import java.util.Base64;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.bouncycastle.asn1.x500.RDN;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.dssclient.http.HttpConnector;
import ru.yandex.direct.dssclient.http.HttpConnectorSettings;
import ru.yandex.direct.dssclient.http.auth.OwnerAuth;
import ru.yandex.direct.dssclient.http.certificates.Certificate;
import ru.yandex.direct.dssclient.http.certificates.SubjectField;
import ru.yandex.direct.dssclient.http.requests.signdocument.Parameters;
import ru.yandex.direct.dssclient.http.requests.signdocument.SignDocumentRequest;
import ru.yandex.direct.dssclient.http.requests.signdocument.Signature;
import ru.yandex.direct.dssclient.http.responses.CertificateResponse;
import ru.yandex.direct.dssclient.pdfstamp.Stamp;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;

/**
 * Клиент к КриптоПро DSS (digital signature service) Яндекса
 * <p>
 * {@see https://www.cryptopro.ru/products/dss}
 * {@see https://github.yandex-team.ru/Billing/dssclient}
 * <p>
 * Текущая реализация сделана только для того, чтобы можно было подписывать PDF-файлы,
 * так что может потребоваться доработка, если надо будет делать с DSS что-то ещё.
 * <p>
 * Как этим пользоваться для подписывания PDF-файлов:
 * <p>
 * 1). создать клиент с авторизационными данными и настройками соединения:
 * <code>
 * client = new DssClient(HttpConnectorSettings.DEFAULT,
 * new DssUserCredentials(username, password),
 * new DssClientCredentials(clientId, clientSecret))
 * </code>
 * <p>
 * 2). опционально: получить из DSS ID сертификата и pinCode:
 * <code>
 * certificates = client.getCertificateList()
 * certificate = certificates.stream().filter(Certificate::isActive).findFirst().get()
 * certificateId = certificate.getId()
 * pinCode = certificate.getPinCode()
 * </code>
 * <p>
 * 3). создать штамп:
 * <code>
 * stamp = Stamp.textStamp(Box.DEFAULT, singletonList(new Text("stamp content")))
 * </code>
 * {@link ru.yandex.direct.dssclient.pdfstamp.Stamp}
 * <p>
 * 4). подписать документ:
 * <code>
 * signedPdfBytes = client.signPdf(pdfBytes, pdfFileName, stamp, certificateId, pinCode)
 * </code>
 */
public class DssClient {
    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();
    private static final Logger LOGGER = LoggerFactory.getLogger(DssClient.class);

    private final HttpConnector httpConnector;

    public DssClient(HttpConnectorSettings httpConnectorSettings,
                     DssUserCredentials userCredentials, DssClientCredentials clientCredentials) {
        OwnerAuth auth = new OwnerAuth(httpConnectorSettings, userCredentials, clientCredentials);
        httpConnector = new HttpConnector(auth, httpConnectorSettings);
    }

    public List<Certificate> getCertificateList() {
        return Stream.of(httpConnector.getRequest("/SignServer/rest/api/certificates", CertificateResponse[].class))
                .map(DssClient::certificateFromApiResponse)
                .collect(toList());
    }

    private static Certificate certificateFromApiResponse(CertificateResponse certificateResponse) {
        Long id = certificateResponse.getId();
        String pinCode = certificateResponse.getStatus().getPinCode();

        byte[] certificateBytes = Base64.getDecoder().decode(certificateResponse.getCertificateBase64());
        org.bouncycastle.asn1.x509.Certificate certificateData =
                org.bouncycastle.asn1.x509.Certificate.getInstance(certificateBytes);

        BigInteger certificateSerialNumber = certificateData.getSerialNumber().getValue();

        Map<SubjectField, String> subject = Arrays.stream(certificateData.getSubject().getRDNs())
                .map(RDN::getFirst)
                .filter(typeAndValue -> SubjectField.valueOfCode(typeAndValue.getType().getId()) != null)
                .collect(toMap(
                        typeAndValue -> SubjectField.valueOfCode(typeAndValue.getType().getId()),
                        typeAndValue -> typeAndValue.getValue().toString()));

        CertificateResponse.Status.StatusValue certificateStatus = certificateResponse.getStatus().getValue();
        boolean isActive = certificateStatus == CertificateResponse.Status.StatusValue.ACTIVE
                || certificateStatus == CertificateResponse.Status.StatusValue.OUT_OF_ORDER;

        Date startDate = certificateData.getStartDate().getDate();
        Date endDate = certificateData.getEndDate().getDate();

        return new Certificate(id, pinCode, certificateSerialNumber, isActive, startDate, endDate, subject);
    }

    public byte[] signPdf(byte[] documentContent, String documentName, Stamp stamp, Long certificateId, String pinCode) {
        String content = Base64.getEncoder().encodeToString(documentContent);

        String pdfSignature;
        try {
            pdfSignature = OBJECT_MAPPER.writeValueAsString(stamp.toJsonObject());
        } catch (JsonProcessingException e) {
            throw new DssClientException(e);
        }

        LOGGER.debug("pdfSignature = {}", pdfSignature);

        SignDocumentRequest request = new SignDocumentRequest.Builder()
                .withContent(content)
                .withName(documentName)
                .withSignature(new Signature.Builder()
                        .withCertificateId(certificateId)
                        .withPinCode(pinCode)
                        .withType(3L)
                        .withParameters(new Parameters.Builder()
                                .withPdfFormat(Parameters.PdfFormat.CMS)
                                .withPdfSignatureTemplateId(stamp.getTemplate().getValue())
                                .withPdfSignatureAppearance(Base64.getEncoder().encodeToString(pdfSignature.getBytes()))
                                .build())
                        .build())
                .build();

        String resultBase64 = httpConnector.jsonPostRequest("/SignServer/rest/api/documents", request);

        if (resultBase64.startsWith("\"")) {
            resultBase64 = resultBase64.substring(1);
        }

        if (resultBase64.endsWith("\"")) {
            resultBase64 = resultBase64.substring(0, resultBase64.length() - 1);
        }

        return Base64.getDecoder().decode(resultBase64);
    }
}
