package ru.yandex.direct.dssclient.http;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.net.HttpURLConnection;
import java.nio.charset.StandardCharsets;
import java.util.Map;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.ImmutableMap;
import com.google.common.io.CharStreams;
import org.apache.commons.io.IOUtils;
import org.apache.http.HttpEntity;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.utils.URLEncodedUtils;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.protocol.HTTP;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.dssclient.DssClientException;
import ru.yandex.direct.dssclient.http.auth.Auth;
import ru.yandex.direct.dssclient.http.responses.OauthTokenResponse;

import static java.util.stream.Collectors.toList;

public final class HttpConnector {
    private static final ObjectMapper OBJECT_MAPPER =
            new ObjectMapper()
                    .configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);

    private static final Logger LOGGER = LoggerFactory.getLogger(HttpConnector.class);

    private final Auth auth;
    private final String baseUrl;
    private final int timeoutInMs;

    public HttpConnector(Auth auth) {
        this(auth, HttpConnectorSettings.DEFAULT);
    }

    public HttpConnector(Auth auth, HttpConnectorSettings settings) {
        this.auth = auth;
        this.baseUrl = settings.getBaseUrl();
        this.timeoutInMs = settings.getTimeoutInMs();
    }

    private <T> T urlEncodedPostRequest(String endpoint, Map<String, String> data, Class<T> responseType) {
        CloseableHttpClient httpClient = getHttpClient();

        try {
            String fullUrl = baseUrl + endpoint;

            LOGGER.debug("fullUrl = {}", fullUrl);
            HttpPost request = new HttpPost(fullUrl);

            String encodedParams = URLEncodedUtils.format(data.entrySet().stream()
                    .map(entry -> new BasicNameValuePair(entry.getKey(), entry.getValue()))
                    .collect(toList()), HTTP.DEF_CONTENT_CHARSET);

            LOGGER.debug("entity = {}", encodedParams);
            request.setEntity(new StringEntity(encodedParams));

            request.setHeader("User-Agent", "ru.yandex.direct.dssclient/1.0");
            request.setHeader("Authorization", auth.getAuthHeader());

            String responseBody = executeRequest(httpClient, request);

            return OBJECT_MAPPER.readValue(responseBody, responseType);
        } catch (IOException e) {
            throw new DssClientException(e);
        }

    }

    public String jsonPostRequest(String endpoint, Object data) {
        CloseableHttpClient httpClient = getHttpClient();

        try {
            String fullUrl = baseUrl + endpoint;

            LOGGER.debug("fullUrl = {}", fullUrl);
            HttpPost request = new HttpPost(fullUrl);

            String encodedParams = OBJECT_MAPPER.writeValueAsString(data);

            LOGGER.debug("entity = {}", encodedParams);
            request.setEntity(new StringEntity(encodedParams, ContentType.APPLICATION_JSON));

            request.setHeader("User-Agent", "ru.yandex.direct.dssclient/1.0");
            request.setHeader("Authorization", auth.getAuthHeader());

            String responseBody = executeRequest(httpClient, request);

            return responseBody;
        } catch (IOException e) {
            throw new DssClientException(e);
        }

    }

    public <T> T getRequest(String endpoint, Class<T> responseType) {
        CloseableHttpClient httpClient = getHttpClient();

        try {
            String fullUrl = baseUrl + endpoint;

            LOGGER.debug("fullUrl = {}", fullUrl);
            HttpGet request = new HttpGet(fullUrl);

            request.setHeader("User-Agent", "ru.yandex.direct.dssclient/1.0");
            request.setHeader("Authorization", auth.getAuthHeader());

            String responseBody = executeRequest(httpClient, request);

            return OBJECT_MAPPER.readValue(responseBody, responseType);
        } catch (IOException e) {
            throw new DssClientException(e);
        }

    }

    private static String executeRequest(CloseableHttpClient httpClient, HttpUriRequest request) throws IOException {
        CloseableHttpResponse httpResponse = httpClient.execute(request);
        LOGGER.debug("response status = {} {}",
                httpResponse.getStatusLine().getStatusCode(),
                httpResponse.getStatusLine().getReasonPhrase());

        if (httpResponse.getStatusLine().getStatusCode() != HttpURLConnection.HTTP_OK) {
            throw new DssClientException(String.format("invalid response status: %s %s; entity = \"%s\"",
                    httpResponse.getStatusLine().getStatusCode(),
                    httpResponse.getStatusLine().getReasonPhrase(),
                    IOUtils.toString(httpResponse.getEntity().getContent(), StandardCharsets.UTF_8)));
        }

        String responseBody = readEntityContent(httpResponse.getEntity());
        LOGGER.debug("responseBody = {}", responseBody);

        return responseBody;
    }

    private static String readEntityContent(HttpEntity responseEntity) throws IOException {
        String responseBody;
        try (Reader reader = new InputStreamReader(responseEntity.getContent())) {
            responseBody = CharStreams.toString(reader);
        }
        return responseBody;
    }

    private CloseableHttpClient getHttpClient() {
        RequestConfig.Builder requestConfigBuilder = RequestConfig.copy(RequestConfig.DEFAULT);

        requestConfigBuilder.setConnectionRequestTimeout(timeoutInMs);
        requestConfigBuilder.setConnectTimeout(timeoutInMs);
        requestConfigBuilder.setSocketTimeout(timeoutInMs);

        HttpClientBuilder httpClientBuilder = HttpClients.custom();

        httpClientBuilder.setDefaultRequestConfig(requestConfigBuilder.build());

        return httpClientBuilder.build();
    }

    public OauthTokenResponse getOauthToken(String clientId, String username, String password) {
        return urlEncodedPostRequest("/STS/oauth/token",
                ImmutableMap.<String, String>builder()
                        .put("grant_type", "password")
                        .put("client_id", clientId)
                        .put("scope", "dss")
                        .put("username", username)
                        .put("password", password)
                        .put("resource", baseUrl + "/SignServer/SignServiceR.svc/token/transport/nosc")
                        .build(),
                OauthTokenResponse.class);
    }

}
