package ru.yandex.direct.dssclient.pdfstamp;

import java.util.List;

import com.google.common.collect.ImmutableMap;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.stream.Collectors.toList;

/**
 * Параметры штампа в PDF-документе. Штамп ставится на определённой странице документа (page) в определённом
 * месте в прямоугольнике (box) и может быть одного из трёх форматов:
 * <p>
 * 1). textStamp: несколько текстовых надписей (textContent)
 * 2). textWithIconStamp: несколько текстовых надписей и изображение (textContent, icon)
 * 3). imageStamp: изображение (foregroundImage)
 * <p>
 * В случае формата "текст + изображение" текущая реализация запрещает указывать цвет фона прямоугольника
 * (box.backgroundColor), потому что, как показали эксперименты, с текущей реализацией DSS такая комбинация
 * не работает.
 */
public class Stamp {
    public static final int DEFAULT_PAGE = 1;

    private final int page;
    private final Box box;
    private final List<Text> textContent;
    private final Template template;
    private final Icon icon;
    private final Image foregroundImage;

    public enum Template {
        TEXT_ONLY(1),
        TEXT_WITH_ICON(2),
        IMAGE(3);

        private final int value;

        Template(int value) {
            this.value = value;
        }

        public int getValue() {
            return value;
        }
    }

    public static Stamp textStamp(Box box, List<Text> textContent) {
        return textStamp(DEFAULT_PAGE, box, textContent);
    }

    public static Stamp textStamp(int page, Box box, List<Text> textContent) {
        return new Stamp(page, box, textContent, Template.TEXT_ONLY, null, null);
    }

    public static Stamp textWithIconStamp(Box box, List<Text> textContent, Icon icon) {
        return textWithIconStamp(DEFAULT_PAGE, box, textContent, icon);
    }

    public static Stamp textWithIconStamp(int page, Box box, List<Text> textContent, Icon icon) {
        return new Stamp(page, box, textContent, Template.TEXT_WITH_ICON, icon, null);
    }

    public static Stamp imageStamp(Box box, Image foregroundImage) {
        return imageStamp(DEFAULT_PAGE, box, foregroundImage);
    }

    public static Stamp imageStamp(int page, Box box, Image foregroundImage) {
        return new Stamp(page, box, null, Template.IMAGE, null, foregroundImage);
    }

    private Stamp(int page, Box box, List<Text> textContent, Template template, Icon icon,
                  Image foregroundImage) {
        this.page = page;
        this.box = box;
        this.textContent = textContent;

        switch (template) {
            case TEXT_ONLY:
                checkNotNull(textContent);
                checkArgument(icon == null);
                checkArgument(foregroundImage == null);
                break;
            case TEXT_WITH_ICON:
                checkNotNull(textContent);
                checkNotNull(icon);
                checkArgument(box.getBackgroundColor() == null);
                checkArgument(foregroundImage == null);
                break;
            case IMAGE:
                checkArgument(textContent == null);
                checkArgument(icon == null);
                checkNotNull(foregroundImage);
                break;
        }

        this.template = template;
        this.icon = icon;
        this.foregroundImage = foregroundImage;
    }

    public int getPage() {
        return page;
    }

    public Box getBox() {
        return box;
    }

    public List<Text> getTextContent() {
        return textContent;
    }

    public Template getTemplate() {
        return template;
    }

    public Icon getIcon() {
        return icon;
    }

    public Image getForegroundImage() {
        return foregroundImage;
    }

    public Object toJsonObject() {
        ImmutableMap.Builder<String, Object> resultBuilder = ImmutableMap.<String, Object>builder()
                .put("Page", page)
                .put("TemplateId", template.value)
                .put("Rect", box.toJsonObject());

        if (textContent != null) {
            resultBuilder.put("Content", textContent.stream().map(Text::toJsonObject).collect(toList()));
        }

        if (icon != null) {
            resultBuilder.put("Icon", icon.toJsonObject());
        }

        if (foregroundImage != null) {
            resultBuilder.put("Background", foregroundImage.toJsonObject());
        }

        return resultBuilder.build();
    }
}
