package ru.yandex.direct.excelmapper;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;

import org.apache.commons.collections4.CollectionUtils;

import ru.yandex.direct.excelmapper.mappers.EnumExcelMapper;
import ru.yandex.direct.excelmapper.mappers.HyperlinkExcelMapper;
import ru.yandex.direct.excelmapper.mappers.HyperlinkOrStringMapper;
import ru.yandex.direct.excelmapper.mappers.IntegerExcelMapper;
import ru.yandex.direct.excelmapper.mappers.IntegerZeroToNullExcelMapper;
import ru.yandex.direct.excelmapper.mappers.ListExcelMapper;
import ru.yandex.direct.excelmapper.mappers.LocalDateMapper;
import ru.yandex.direct.excelmapper.mappers.LocalDateTimeMapper;
import ru.yandex.direct.excelmapper.mappers.LongExcelMapper;
import ru.yandex.direct.excelmapper.mappers.MaybeEmptyExcelMapper;
import ru.yandex.direct.excelmapper.mappers.ObjectExcelMapper;
import ru.yandex.direct.excelmapper.mappers.PredicateCheckingExcelMapper;
import ru.yandex.direct.excelmapper.mappers.SetExcelMapper;
import ru.yandex.direct.excelmapper.mappers.StringExcelMapper;
import ru.yandex.direct.excelmapper.mappers.YesNoBooleanExcelMapper;
import ru.yandex.direct.model.Model;

public class ExcelMappers {
    private ExcelMappers() {
    }

    public static ExcelMapper<String> stringMapper(String title) {
        return new StringExcelMapper(title);
    }

    public static ExcelMapper<String> maybeStringMapper(String title) {
        return new MaybeEmptyExcelMapper<>(new StringExcelMapper(title));
    }

    public static ExcelMapper<Integer> intMapper(String title) {
        return new IntegerExcelMapper(title);
    }

    public static ExcelMapper<Integer> maybeIntMapper(String title) {
        return new MaybeEmptyExcelMapper<>(new IntegerExcelMapper(title));
    }

    public static ExcelMapper<Integer> intZeroToNullMapper(String title) {
        return new IntegerZeroToNullExcelMapper(title);
    }

    public static ExcelMapper<Integer> maybeIntZeroToNullMapper(String title) {
        return new MaybeEmptyExcelMapper<>(new IntegerZeroToNullExcelMapper(title));
    }

    public static ExcelMapper<Long> longMapper(String title) {
        return new LongExcelMapper(title);
    }

    public static ExcelMapper<Long> maybeLongMapper(String title) {
        return new MaybeEmptyExcelMapper<>(new LongExcelMapper(title));
    }

    public static ExcelMapper<Boolean> yesNoBooleanMapper(String title) {
        return new YesNoBooleanExcelMapper(title);
    }

    public static ExcelMapper<Boolean> maybeYesNoBooleanMapper(String title) {
        return new MaybeEmptyExcelMapper<>(new YesNoBooleanExcelMapper(title));
    }

    public static <T extends Enum<T>> ExcelMapper<T> enumMapper(String title, Class<T> enumClass) {
        return new EnumExcelMapper<>(title, enumClass);
    }

    public static <T extends Enum<T>> ExcelMapper<T> enumMapper(String title,
                                                                Function<T, String> writeMapper,
                                                                Function<String, T> readMapper) {
        return new EnumExcelMapper<>(title, writeMapper, readMapper);
    }

    public static <T extends Enum<T>> ExcelMapper<T> maybeEnumMapper(String title, Class<T> enumClass) {
        return new MaybeEmptyExcelMapper<>(new EnumExcelMapper<>(title, enumClass));
    }

    public static <T extends Enum<T>> ExcelMapper<T> maybeEnumMapper(String title,
                                                                     Function<T, String> writeMapper,
                                                                     Function<String, T> readMapper) {
        return new MaybeEmptyExcelMapper<>(new EnumExcelMapper<>(title, writeMapper, readMapper));
    }

    public static <T> ExcelMapper<List<T>> listMapper(ExcelMapper<T> itemMapper) {
        return listMapper(itemMapper, false);
    }

    public static <T> ExcelMapper<List<T>> listMapper(ExcelMapper<T> itemMapper, boolean addBorderBottomPerItem) {
        return new ListExcelMapper<>(itemMapper, addBorderBottomPerItem);
    }

    public static <T> ExcelMapper<List<T>> maybeListMapper(ExcelMapper<T> itemMapper) {
        return maybeListMapper(itemMapper, false);
    }

    public static <T> ExcelMapper<List<T>> maybeListMapper(ExcelMapper<T> itemMapper, boolean addBorderBottomPerItem) {
        return new MaybeEmptyExcelMapper<>(new ListExcelMapper<>(itemMapper, addBorderBottomPerItem),
                CollectionUtils::isEmpty, List::of);
    }

    public static <T> ExcelMapper<Set<T>> setMapper(ExcelMapper<T> itemMapper) {
        return new SetExcelMapper<>(itemMapper);
    }

    public static <T> ExcelMapper<Set<T>> maybeSetMapper(ExcelMapper<T> itemMapper) {
        return new MaybeEmptyExcelMapper<>(new SetExcelMapper<>(itemMapper), CollectionUtils::isEmpty, Set::of);
    }

    public static <X> ObjectExcelMapper.ObjectExcelMapperBuilder<X> objectMapper(Supplier<X> constructor) {
        return ObjectExcelMapper.builder(constructor);
    }

    public static <M extends Model> ObjectExcelMapper.ModelExcelMapperBuilder<M> modelMapper(Supplier<M> constructor) {
        return ObjectExcelMapper.builderModel(constructor);
    }

    public static ExcelMapper<LocalDate> localDateMapper(String title) {
        return new LocalDateMapper(title);
    }

    public static ExcelMapper<LocalDate> maybeLocalDateMapper(String title) {
        return new MaybeEmptyExcelMapper<>(new LocalDateMapper(title));
    }

    public static ExcelMapper<LocalDateTime> localDateTimeMapper(String title) {
        return new LocalDateTimeMapper(title);
    }

    public static ExcelMapper<LocalDateTime> maybeLocalDateTimeMapper(String title) {
        return new MaybeEmptyExcelMapper<>(new LocalDateTimeMapper(title));
    }

    public static ExcelMapper<String> hyperlinkExcelMapper(String title) {
        return new HyperlinkExcelMapper(title);
    }

    public static ExcelMapper<String> hyperlinkOrStringMapper(String title) {
        return new HyperlinkOrStringMapper(title);
    }

    public static ExcelMapper<String> maybeHyperlinkOrStringMapper(String title) {
        return new MaybeEmptyExcelMapper<>(new HyperlinkOrStringMapper(title));
    }

    public static <T> ExcelMapper<T> predicateCheckingMapper(ExcelMapper<T> mapper, Predicate<T> predicate) {
        return new PredicateCheckingExcelMapper<>(mapper, predicate);
    }

    public static <T> ExcelMapper<T> maybePredicateCheckingMapper(ExcelMapper<T> mapper, Predicate<T> predicate) {
        return new MaybeEmptyExcelMapper<>(predicateCheckingMapper(mapper, predicate));
    }
}
