package ru.yandex.direct.excelmapper;

import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.apache.poi.ss.usermodel.Cell;

import static com.google.common.base.Preconditions.checkArgument;

public class MapperUtils {
    private MapperUtils() {
    }

    /**
     * Проверяет правда ли, что все ячейки пусты.
     *
     * Если данные какой-то ячейка не соответствуют CellType,
     * бросает исключение {@link SheetRange#reportInvalidCellDataFormat}.
     * */
    public static boolean allCellsAreEmpty(SheetRange sheetRange, int height, int width, List<String> columns) {
        for (int row = 0; row < height; row++) {
            for (int col = 0; col < width; col++) {
                String value = tryReadCellValueOrReportCantRead(sheetRange, row, col, columns);

                if (StringUtils.isNotBlank(value)) {
                    return false;
                }
            }
        }
        return true;
    }

    /**
     * Читает значение из ячейки. Если данные ячейки не соответсвуют формату, то бросает
     * исключение {@link SheetRange#reportInvalidCellDataFormat}
     *
     * @param sheetRange   область из которой пытаемся прочитать ячейку
     * @param row          номер строки
     * @param col          номер столбца
     * @param columnTitles заголовки столбцов области из которой пытаемся прочитать ячейку
     */
    public static String tryReadCellValueOrReportCantRead(SheetRange sheetRange, int row, int col,
                                                          List<String> columnTitles) {
        checkArgument(col < columnTitles.size(), "column title not found");
        try {
            Cell cell = sheetRange.getCell(row, col);
            switch (cell.getCellTypeEnum()) {
                case FORMULA:
                    return cell.getCellFormula().trim();
                default:
                    return cell.getStringCellValue().trim();
            }
        } catch (IllegalStateException e) {
            sheetRange.reportInvalidCellDataFormat(columnTitles.subList(col, col + 1), row, col);
        }
        // недостижимое место
        throw new IllegalStateException();
    }

    /**
     * Читает значение из ячейки. Если формат ячейки не является строкой, то бросает
     * исключение {@link SheetRange#reportInvalidCellDataFormat}
     *
     * @param sheetRange   область из которой пытаемся прочитать ячейку
     * @param row          номер строки
     * @param col          номер столбца
     * @param columnTitles заголовки столбцов области из которой пытаемся прочитать ячейку
     */
    public static String tryReadStringCellValueOrReportCantRead(SheetRange sheetRange, int row, int col,
                                                                List<String> columnTitles) {
        checkArgument(col < columnTitles.size(), "column title not found");
        try {
            return sheetRange.getCell(row, col).getStringCellValue().trim();
        } catch (IllegalStateException e) {
            sheetRange.reportInvalidCellDataFormat(columnTitles.subList(col, col + 1), row, col);
        }
        // недостижимое место
        throw new IllegalStateException();
    }

    /**
     * Читает значение из ячейки. Если формат ячейки не является формулой, то бросает
     * исключение {@link SheetRange#reportInvalidCellDataFormat}
     *
     * @param sheetRange   область из которой пытаемся прочитать ячейку
     * @param row          номер строки
     * @param col          номер столбца
     * @param columnTitles заголовки столбцов области из которой пытаемся прочитать ячейку
     */
    public static String tryReadFormulaCellValueOrReportCantRead(SheetRange sheetRange, int row, int col,
                                                                List<String> columnTitles) {
        checkArgument(col < columnTitles.size(), "column title not found");
        try {
            return sheetRange.getCell(row, col).getCellFormula().trim();
        } catch (IllegalStateException e) {
            sheetRange.reportInvalidCellDataFormat(columnTitles.subList(col, col + 1), row, col);
        }
        // недостижимое место
        throw new IllegalStateException();
    }
}
