package ru.yandex.direct.excelmapper;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import one.util.streamex.StreamEx;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;

import ru.yandex.direct.excelmapper.exceptions.CantReadEmptyException;
import ru.yandex.direct.excelmapper.exceptions.CantReadFormatException;
import ru.yandex.direct.excelmapper.exceptions.CantReadRangeTooSmallException;
import ru.yandex.direct.excelmapper.exceptions.CantReadUnexpectedDataException;
import ru.yandex.direct.excelmapper.exceptions.InvalidCellDataFormatException;

import static com.google.common.base.Preconditions.checkArgument;

public class SheetRangeImpl implements SheetRange {
    private final Sheet sheet;

    private final int firstRow;
    private final int firstCol;
    private final int height;

    private final Set<Integer> changedColumns = new HashSet<>();
    private final List<SheetRange> innerRanges = new ArrayList<>();

    SheetRangeImpl(Sheet sheet, int firstRow, int firstCol, int height) {
        this.sheet = sheet;
        this.firstRow = firstRow;
        this.firstCol = firstCol;
        this.height = height;
    }

    public SheetRangeImpl(Sheet sheet, int firstRow, int firstCol, HeightMode heightMode) {
        this.sheet = sheet;
        this.firstRow = firstRow;
        this.firstCol = firstCol;
        this.height = calcHeight(firstRow, heightMode);
    }

    @Override
    public int getHeight() {
        return height;
    }

    @Override
    public SheetRange makeSubRange(int offsetRows, int offsetCols, int subRangeHeight) {
        checkArgument(offsetRows + subRangeHeight <= height);
        SheetRangeImpl sheetRange = new SheetRangeImpl(sheet, firstRow + offsetRows, firstCol + offsetCols,
                subRangeHeight);
        innerRanges.add(sheetRange);
        return sheetRange;
    }

    @Override
    public SheetRange makeSubRange(int offsetRows, int offsetCols, HeightMode heightMode) {
        var newFirstRow = firstRow + offsetCols;
        int newHeight = calcHeight(newFirstRow, heightMode);
        SheetRangeImpl sheetRange = new SheetRangeImpl(sheet, firstRow + offsetRows, firstCol + offsetCols,
                Integer.min(height, newHeight));
        innerRanges.add(sheetRange);
        return sheetRange;
    }

    @Override
    public SheetRange makeSubRange(int offsetRows, int offsetCols) {
        checkArgument(offsetRows < height);
        return makeSubRange(offsetRows, offsetCols, height - offsetRows);
    }

    @Override
    public boolean isOffsetRowsInRange(int offsetRows) {
        return offsetRows < height;
    }

    @Override
    public Cell getCell(int row, int col) {
        checkArgument(row < height);
        return getAbsoluteCell(this.firstRow + row, this.firstCol + col);
    }

    @Override
    public Cell getCellForWrite(int row, int col) {
        Cell cell = getCell(row, col);
        changedColumns.add(cell.getColumnIndex());
        return cell;
    }

    @Override
    public Set<Integer> getColumnsWithNotEmptyValue() {
        return StreamEx.of(innerRanges)
                .flatMap(sheetRange -> sheetRange.getColumnsWithNotEmptyValue().stream())
                .append(changedColumns)
                .toSet();
    }

    @Override
    public void reportInvalidCellDataFormat(List<String> columns, int offsetRows, int offsetCols) {
        throw new InvalidCellDataFormatException(columns, this.firstRow + offsetRows, this.firstCol + offsetCols);
    }

    @Override
    public void reportCantReadEmpty(List<String> columns, int offsetRows, int offsetCols) {
        throw new CantReadEmptyException(columns, this.firstRow + offsetRows, this.firstCol + offsetCols);
    }

    @Override
    public void reportCantReadFormat(List<String> columns, int offsetRows, int offsetCols) {
        throw new CantReadFormatException(columns, this.firstRow + offsetRows, this.firstCol + offsetCols);
    }

    @Override
    public void reportCantReadRangeMismatch(List<String> columns, int offsetRows, int offsetCols) {
        throw new CantReadRangeTooSmallException(columns, this.firstRow + offsetRows, this.firstCol + offsetCols);

    }

    @Override
    public void reportCantReadUnexpectedData(List<String> columns, int offsetRows, int offsetCols) {
        throw new CantReadUnexpectedDataException(columns, this.firstRow + offsetRows, this.firstCol + offsetCols);
    }

    private Row getAbsoluteRow(int rowNum) {
        if (sheet.getRow(rowNum) != null) {
            return sheet.getRow(rowNum);
        }
        return sheet.createRow(rowNum);
    }

    private Cell getAbsoluteCell(int rowNum, int colNum) {
        Row row = getAbsoluteRow(rowNum);
        if (row.getCell(colNum) != null) {
            return row.getCell(colNum);
        }
        return row.createCell(colNum);
    }

    private int calcHeight(int firstRow, HeightMode heightMode) {
        int newHeight;
        if (heightMode == HeightMode.AUTODETECT_HEIGHT) {
            newHeight = sheet.getLastRowNum() + 1 - firstRow;
        } else if (heightMode == HeightMode.MAX_HEIGHT) {
            newHeight = sheet.getWorkbook().getSpreadsheetVersion().getMaxRows();
        } else {
            throw new IllegalArgumentException(String.format("Unknown value of heightMode %s", heightMode));
        }
        return newHeight;
    }
}
