package ru.yandex.direct.excelmapper.mappers;

import java.util.Collection;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.poi.ss.usermodel.BorderStyle;
import org.apache.poi.xssf.usermodel.XSSFCellStyle;

import ru.yandex.direct.excelmapper.ExcelMapper;
import ru.yandex.direct.excelmapper.Height;
import ru.yandex.direct.excelmapper.MapperMeta;
import ru.yandex.direct.excelmapper.MapperUtils;
import ru.yandex.direct.excelmapper.ReadResult;
import ru.yandex.direct.excelmapper.SheetRange;
import ru.yandex.direct.excelmapper.exceptions.CantWriteEmptyException;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;

/**
 * Абстрактный класс маппера для коллекций
 */
@ParametersAreNonnullByDefault
abstract class AbstractCollectionExcelMapper<T, C extends Collection<T>> implements ExcelMapper<C> {

    private final ExcelMapper<T> itemMapper;
    private final Supplier<C> collectionSupplier;
    private final MapperMeta meta;
    private final boolean addBorderBottomPerItem;

    public AbstractCollectionExcelMapper(ExcelMapper<T> itemMapper, Supplier<C> collectionSupplier,
                                         boolean addBorderBottomPerItem) {
        checkArgument(!itemMapper.canBeEmpty(), "Can't create collection mapper if item mapper value can be empty");
        this.itemMapper = itemMapper;
        this.collectionSupplier = collectionSupplier;
        this.addBorderBottomPerItem = addBorderBottomPerItem;

        MapperMeta itemMapperMeta = itemMapper.getMeta();
        // подразумевается, что для всех наследников canBeEmpty() == false
        this.meta = new MapperMeta(itemMapperMeta.getColumns(), Height.dynamic(), itemMapperMeta.getRequiredColumns());
    }

    @Override
    public MapperMeta getMeta() {
        return meta;
    }

    @Override
    public final boolean canBeEmpty() {
        return false;
    }

    @Override
    public int write(SheetRange sheetRange, @Nullable C items) {
        if (CollectionUtils.isEmpty(items)) {
            throw new CantWriteEmptyException(getMeta().getColumns());
        }
        int verticalOffset = 0;
        for (T item : items) {
            var writtenRows = itemMapper.write(sheetRange.makeSubRange(verticalOffset, 0), item);
            verticalOffset += writtenRows;

            if (addBorderBottomPerItem) {
                for (int column = 0; column < getMeta().getWidth(); column++) {
                    for (int row = verticalOffset - writtenRows; row < verticalOffset; row++) {
                        sheetRange.getCell(row, column).setCellStyle(sheetRange.getCell(row, column).getCellStyle());
                    }

                    setBorderCellStyle(sheetRange, verticalOffset - 1, column);
                }
            }
        }
        return verticalOffset;
    }

    private static void setBorderCellStyle(SheetRange sheetRange, int row, int column) {
        XSSFCellStyle cellStyle = (XSSFCellStyle) sheetRange.getCell(row, column).getCellStyle();
        XSSFCellStyle newCellStyle = (XSSFCellStyle) cellStyle.clone();
        newCellStyle.setBorderBottom(BorderStyle.THICK);
        sheetRange.getCell(row, column).setCellStyle(newCellStyle);
    }

    @Override
    public ReadResult<C> read(SheetRange sheetRange) {
        int verticalOffsetStart = 0;

        var result = collectionSupplier.get();

        Height itemMapperHeight = itemMapper.getMeta().getHeight();
        do {
            int areaHeight = calcItemAreaHeight(sheetRange, verticalOffsetStart, itemMapperHeight);
            SheetRange itemRange = sheetRange.makeSubRange(verticalOffsetStart, 0, areaHeight);
            ReadResult<T> read = itemMapper.read(itemRange);
            checkState(read.getReadRows() == areaHeight);
            result.add(read.getValue());

            verticalOffsetStart += areaHeight;
        } while (sheetRange.isOffsetRowsInRange(verticalOffsetStart)
                && itemMapper.canStartReading(sheetRange.makeSubRange(verticalOffsetStart, 0)));

        while (sheetRange.isOffsetRowsInRange(verticalOffsetStart)) {
            SheetRange subRange = sheetRange.makeSubRange(verticalOffsetStart, 0);
            if (!MapperUtils.allCellsAreEmpty(subRange, 1, getMeta().getWidth(), getMeta().getColumns())) {
                sheetRange.reportCantReadUnexpectedData(getMeta().getColumns(), verticalOffsetStart, 0);
            }
            verticalOffsetStart += 1;
        }
        return new ReadResult<>(result, verticalOffsetStart);
    }

    private int calcItemAreaHeight(SheetRange sheetRange, int verticalOffsetStart, Height itemMapperHeight) {
        int areaHeight = 1;
        if (itemMapperHeight.isFixed()) {
            areaHeight = itemMapperHeight.getValue();
        } else {
            // ищем, где заканчивается область чтения элемента
            while (sheetRange.isOffsetRowsInRange(verticalOffsetStart + areaHeight)
                    && !itemMapper.canStartReading(sheetRange.makeSubRange(verticalOffsetStart + areaHeight, 0))) {
                areaHeight += 1;
            }
        }
        return areaHeight;
    }

    @Override
    public boolean canStartReading(SheetRange sheetRange) {
        return itemMapper.canStartReading(sheetRange);
    }
}
