package ru.yandex.direct.excelmapper.mappers;

import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;

import org.apache.commons.lang3.StringUtils;
import org.apache.poi.ss.usermodel.Cell;

import ru.yandex.direct.excelmapper.ExcelMapper;
import ru.yandex.direct.excelmapper.Height;
import ru.yandex.direct.excelmapper.MapperMeta;
import ru.yandex.direct.excelmapper.MapperUtils;
import ru.yandex.direct.excelmapper.ReadResult;
import ru.yandex.direct.excelmapper.SheetRange;
import ru.yandex.direct.excelmapper.exceptions.CantWriteEmptyException;

/**
 * Абстрактный класс для записи простых типов и формул в ячейку.
 * */
public abstract class AbstractSingleCellMapper<T> implements ExcelMapper<T> {

    private static final int HEIGHT = 1;
    protected static final int ROW = 0;
    protected static final int COL = 0;

    private final MapperMeta meta;

    public AbstractSingleCellMapper(String title) {
        // подразумевается, что для всех наследников canBeEmpty() == false
        this.meta = new MapperMeta(List.of(title), Height.fixed(HEIGHT), Set.of(title));
    }

    protected abstract String convertValueToWrite(T value);

    protected abstract T convertValueToRead(String value);

    /**
     * Определяет в ячейке формула или другой тип.
     * */
    protected abstract boolean isFormula(String value);

    @Override
    public MapperMeta getMeta() {
        return meta;
    }

    @Override
    public final boolean canBeEmpty() {
        return false;
    }

    @Override
    public int write(SheetRange sheetRange, @Nullable T value) {
        if (value == null) {
            throw new CantWriteEmptyException(getMeta().getColumns());
        }

        String valueToWrite = convertValueToWrite(value);
        if (isFormula(valueToWrite)) {
            sheetRange.getCellForWrite(ROW, COL).setCellFormula(valueToWrite);
        } else {
            sheetRange.getCellForWrite(ROW, COL).setCellValue(valueToWrite);
        }
        return HEIGHT;
    }

    @Override
    public ReadResult<T> read(SheetRange sheetRange) {
        if (sheetRange.getHeight() != HEIGHT) {
            sheetRange.reportCantReadRangeMismatch(getMeta().getColumns(), ROW, COL);
        }
        Cell cell = sheetRange.getCell(ROW, COL);
        String stringCellValue;
        switch (cell.getCellTypeEnum()) {
            case FORMULA:
                stringCellValue = cell.getCellFormula();
                break;
            default:
                stringCellValue = cell.getStringCellValue();
        }

        if (StringUtils.isNotBlank(stringCellValue)) {
            T value = convertValueToRead(stringCellValue);
            return new ReadResult<>(value, HEIGHT);
        } else {
            sheetRange.reportCantReadEmpty(getMeta().getColumns(), ROW, COL);
        }
        // недостижимое место
        throw new IllegalStateException();
    }

    @Override
    public boolean canStartReading(SheetRange sheetRange) {
        String value = tryReadFormulaCellValue(sheetRange);
        return StringUtils.isNotBlank(value);
    }

    private String tryReadFormulaCellValue(SheetRange sheetRange) {
        return MapperUtils.tryReadCellValueOrReportCantRead(sheetRange, ROW, COL, meta.getColumns());
    }
}
