package ru.yandex.direct.excelmapper.mappers;


import java.util.Objects;
import java.util.function.Predicate;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.excelmapper.ExcelMapper;
import ru.yandex.direct.excelmapper.Height;
import ru.yandex.direct.excelmapper.MapperMeta;
import ru.yandex.direct.excelmapper.MapperUtils;
import ru.yandex.direct.excelmapper.ReadResult;
import ru.yandex.direct.excelmapper.SheetRange;

import static java.util.Collections.emptySet;

@ParametersAreNonnullByDefault
public class MaybeEmptyExcelMapper<T> implements ExcelMapper<T> {
    private final MapperMeta meta;
    private final ExcelMapper<T> innerMapper;
    private Predicate<T> isEmptyChecker;
    private Supplier<T> emptyValueSupplier;

    public MaybeEmptyExcelMapper(ExcelMapper<T> innerMapper, Predicate<T> isEmptyChecker,
                                 Supplier<T> emptyValueSupplier) {
        MapperMeta innerMapperMeta = innerMapper.getMeta();
        this.meta = new MapperMeta(innerMapperMeta.getColumns(), innerMapperMeta.getHeight(), emptySet());
        this.innerMapper = innerMapper;
        this.isEmptyChecker = isEmptyChecker;
        this.emptyValueSupplier = emptyValueSupplier;
    }

    public MaybeEmptyExcelMapper(ExcelMapper<T> innerMapper) {
        this(innerMapper, Objects::isNull, () -> null);
    }

    @Override
    public MapperMeta getMeta() {
        return meta;
    }

    private boolean isEmpty(@Nullable T value) {
        return isEmptyChecker.test(value);
    }

    @Nullable
    private T getEmptyValue() {
        return emptyValueSupplier.get();
    }

    @Override
    public boolean canBeEmpty() {
        return true;
    }

    @Override
    public int write(SheetRange sheetRange, @Nullable T value) {
        if (isEmpty(value)) {
            int width = getMeta().getWidth();
            int height = getMeta().getHeight().isFixed() ? getMeta().getHeight().getValue() : 1;
            writeEmptyCells(sheetRange, height, width);
            return height;
        } else {
            return innerMapper.write(sheetRange, value);
        }
    }

    @Override
    public ReadResult<T> read(SheetRange sheetRange) {
        Height height = getMeta().getHeight();
        int rangeHeight = sheetRange.getHeight();
        if (height.isFixed() && rangeHeight != height.getValue()) {
            sheetRange.reportCantReadRangeMismatch(getMeta().getColumns(), 0, 0);
        }
        int width = getMeta().getWidth();
        if (MapperUtils.allCellsAreEmpty(sheetRange, rangeHeight, width, getMeta().getColumns())) {
            return new ReadResult<>(getEmptyValue(), rangeHeight);
        } else {
            return innerMapper.read(sheetRange);
        }
    }

    @Override
    public boolean canStartReading(SheetRange sheetRange) {
        int width = getMeta().getWidth();
        int height = getMeta().getHeight().isFixed() ? getMeta().getHeight().getValue() : 1;
        if (MapperUtils.allCellsAreEmpty(sheetRange, height, width, getMeta().getColumns())) {
            return true;
        }
        return innerMapper.canStartReading(sheetRange);
    }

    private void writeEmptyCells(SheetRange sheetRange, int height, int width) {
        for (int row = 0; row < height; row++) {
            for (int col = 0; col < width; col++) {
                sheetRange.getCell(row, col).setCellValue("");
            }
        }
    }
}
