package ru.yandex.direct.excelmapper.mappers;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.Function;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.excelmapper.ExcelMapper;
import ru.yandex.direct.excelmapper.Height;
import ru.yandex.direct.excelmapper.MapperMeta;
import ru.yandex.direct.excelmapper.MapperUtils;
import ru.yandex.direct.excelmapper.ReadResult;
import ru.yandex.direct.excelmapper.SheetRange;
import ru.yandex.direct.excelmapper.exceptions.CantWriteEmptyException;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;
import static ru.yandex.direct.utils.FunctionalUtils.flatMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public class ObjectExcelMapper<O> implements ExcelMapper<O> {
    private final MapperMeta meta;
    private final Supplier<O> constructor;
    private final List<FieldAdapter<O, ?>> fieldAdapters;

    private ObjectExcelMapper(Supplier<O> constructor, List<FieldAdapter<O, ?>> fieldAdapters) {
        checkArgument(!fieldAdapters.isEmpty(), "At least one field must be present");
        checkArgument(hasFieldWithFixedHeight(fieldAdapters), "Can't create mapper if all fields has dynamic height");
        this.constructor = constructor;
        this.fieldAdapters = fieldAdapters;
        this.meta = calcMeta(fieldAdapters);
    }

    public static <X> ObjectExcelMapperBuilder<X> builder(Supplier<X> constructor) {
        return new ObjectExcelMapperBuilder<>(constructor);
    }

    public static <M extends Model> ModelExcelMapperBuilder<M> builderModel(Supplier<M> constructor) {
        return new ModelExcelMapperBuilder<>(constructor);
    }

    @Override
    public MapperMeta getMeta() {
        return meta;
    }

    @Override
    public int write(SheetRange sheetRange, @Nullable O value) {
        if (value == null) {
            throw new CantWriteEmptyException(getMeta().getColumns());
        }
        int col = 0;
        int rows = 0;
        for (var adapter : fieldAdapters) {
            int writtenRows = adapter.writeFrom(sheetRange.makeSubRange(0, col), value);
            if (rows < writtenRows) {
                rows = writtenRows;
            }
            col += adapter.getWidth();
        }
        return rows;
    }

    @Override
    public ReadResult<O> read(SheetRange sheetRange) {
        O newObject = constructor.get();
        int col = 0;
        int rows = 0;
        for (var adapter : fieldAdapters) {
            final SheetRange sheetRangeForField;
            if (adapter.getHeight().isFixed()) {
                sheetRangeForField = sheetRange.makeSubRange(0, col, adapter.getHeight().getValue());
            } else {
                sheetRangeForField = sheetRange.makeSubRange(0, col);
            }
            int readRows = adapter.readInto(sheetRangeForField, newObject);
            if (rows < readRows) {
                rows = readRows;
            }
            col += adapter.getWidth();
        }
        return new ReadResult<>(newObject, rows);
    }

    /**
     * Можем начать читать объект если:
     * - есть хотя-бы одно непустое поле с фиксированной высотой, которую можем прочесть
     * - ИЛИ можем прочесть все поля и первая строка из sheetRange непустая
     */
    @Override
    public boolean canStartReading(SheetRange sheetRange) {
        int col = 0;
        boolean allAdaptersCanStartReading = true;
        for (var adapter : fieldAdapters) {
            if (adapter.cannotBeEmptyFieldWithFixedHeight()
                    && adapter.canStartReading(sheetRange.makeSubRange(0, col))) {
                return true;
            }

            if (!adapter.canStartReading(sheetRange.makeSubRange(0, col))) {
                allAdaptersCanStartReading = false;
            }

            col += adapter.getWidth();
        }

        if (MapperUtils.allCellsAreEmpty(sheetRange, 1, col, getMeta().getColumns())) {
            return false;
        }

        return allAdaptersCanStartReading;
    }

    private boolean hasFieldWithFixedHeight(List<FieldAdapter<O, ?>> fieldAdapters) {
        return StreamEx.of(fieldAdapters)
                .map(FieldAdapter::getHeight)
                .anyMatch(Height::isFixed);
    }

    private MapperMeta calcMeta(List<FieldAdapter<O, ?>> fieldAdapters) {
        List<MapperMeta> mappersMeta = mapList(fieldAdapters, fieldAdapter -> fieldAdapter.mapper.getMeta());
        List<String> columns = flatMap(mappersMeta, MapperMeta::getColumns);
        Set<String> requiredColumns = flatMapToSet(mappersMeta, MapperMeta::getRequiredColumns);

        return new MapperMeta(columns, calcMapperHeight(fieldAdapters), requiredColumns);
    }

    private Height calcMapperHeight(List<FieldAdapter<O, ?>> fieldAdapters) {
        List<Height> heights = StreamEx.of(fieldAdapters)
                .map(FieldAdapter::getMapper)
                .map(ExcelMapper::getMeta)
                .map(MapperMeta::getHeight)
                .toList();
        int maxFixedHeight = 1;
        for (Height height : heights) {
            if (height.isDynamic()) {
                return Height.dynamic();
            }
            if (maxFixedHeight < height.getValue()) {
                maxFixedHeight = height.getValue();
            }
        }
        return Height.fixed(maxFixedHeight);
    }

    public static class ObjectExcelMapperBuilder<X> {
        private final Supplier<X> constructor;
        private final List<FieldAdapter<X, ?>> fieldAdapters;

        private ObjectExcelMapperBuilder(Supplier<X> constructor) {
            this.constructor = constructor;
            this.fieldAdapters = new ArrayList<>();
        }

        public <F> ObjectExcelMapperBuilder<X> field(Function<X, F> getter, BiConsumer<X, F> setter,
                                                     ExcelMapper<F> mapper) {
            fieldAdapters.add(new FieldAdapter<>(getter, setter, mapper));
            return this;
        }

        public ObjectExcelMapper<X> build() {
            return new ObjectExcelMapper<>(constructor, fieldAdapters);
        }
    }

    public static class ModelExcelMapperBuilder<M extends Model> extends ObjectExcelMapperBuilder<M> {
        private ModelExcelMapperBuilder(Supplier<M> constructor) {
            super(constructor);
        }

        @Override
        public <F> ModelExcelMapperBuilder<M> field(Function<M, F> getter, BiConsumer<M, F> setter,
                                                    ExcelMapper<F> mapper) {
            super.field(getter, setter, mapper);
            return this;
        }

        public <F> ModelExcelMapperBuilder<M> field(ModelProperty<? super M, F> modelProperty, ExcelMapper<F> mapper) {
            return field(modelProperty::get, modelProperty::set, mapper);
        }

        /**
         * Если значение поля у модели null, то проставим defaultValue
         * Нужно для экспорта новых обязательных полей, по которым в базе нет значения
         */
        public <F> ModelExcelMapperBuilder<M> fieldWithDefaultValueForGetter(ModelProperty<? super M, F> modelProperty,
                                                                             ExcelMapper<F> mapper,
                                                                             F defaultValue) {
            return field(getterWithDefaultValue(modelProperty, defaultValue), modelProperty::set, mapper);
        }


        /**
         * Если в .xls поле пустое, в модель запишем defaultValue
         */
        public <F> ModelExcelMapperBuilder<M> fieldWithDefaultValueForSetter(ModelProperty<? super M, F> modelProperty,
                                                                             ExcelMapper<F> mapper,
                                                                             F defaultValue) {
            return field(modelProperty::get, setterWithDefaultValue(modelProperty, defaultValue), mapper);
        }

        /**
         * Поле для чтения, изменить его нельзя, то есть при импорте в поле будет всегда null
         */
        public <F> ModelExcelMapperBuilder<M> readOnlyField(ModelProperty<? super M, F> modelProperty,
                                                            ExcelMapper<F> mapper) {
            return field(modelProperty::get, this::doNothing, mapper);
        }

        public <F> ModelExcelMapperBuilder<M> readOnlyFieldWithDefaultValue(ModelProperty<? super M, F> modelProperty,
                                                                            ExcelMapper<F> mapper,
                                                                            F defaultValue) {
            return field(getterWithDefaultValue(modelProperty, defaultValue), this::doNothing, mapper);
        }

        private <F> Function<M, F> getterWithDefaultValue(ModelProperty<? super M, F> modelProperty, F defaultValue) {
            return model -> modelProperty.get(model) != null ? modelProperty.get(model) : defaultValue;
        }

        private <F> BiConsumer<M, F> setterWithDefaultValue(ModelProperty<? super M, F> modelProperty, F defaultValue) {
            return (model, value) -> modelProperty.set(model, value != null ? value : defaultValue);
        }

        public <F> void doNothing(M m, F f) {
        }

    }

    private static class FieldAdapter<X, F> {
        private final Function<X, F> getter;
        private final BiConsumer<X, F> setter;
        private final ExcelMapper<F> mapper;

        private FieldAdapter(Function<X, F> getter, BiConsumer<X, F> setter, ExcelMapper<F> mapper) {
            this.getter = getter;
            this.setter = setter;
            this.mapper = mapper;
        }

        private ExcelMapper<F> getMapper() {
            return mapper;
        }

        private int getWidth() {
            return mapper.getMeta().getWidth();
        }

        private int writeFrom(SheetRange sheetRange, X obj) {
            return mapper.write(sheetRange, getter.apply(obj));
        }

        private int readInto(SheetRange sheetRange, X obj) {
            ReadResult<F> readResult = mapper.read(sheetRange);
            setter.accept(obj, readResult.getValue());
            return readResult.getReadRows();
        }

        private Height getHeight() {
            return mapper.getMeta().getHeight();
        }

        private boolean cannotBeEmptyFieldWithFixedHeight() {
            return !getMapper().canBeEmpty() && getHeight().isFixed();
        }

        private boolean canStartReading(SheetRange sheetRange) {
            return mapper.canStartReading(sheetRange);
        }
    }
}
