package ru.yandex.direct.gemini

import org.slf4j.Logger
import org.slf4j.LoggerFactory
import ru.yandex.direct.asynchttp.ParallelFetcherFactory
import ru.yandex.direct.gemini.model.GeminiResponse
import ru.yandex.direct.gemini.model.GeminiSimpleTypeResponse
import ru.yandex.direct.http.smart.core.Smart
import ru.yandex.direct.http.smart.error.ErrorUtils.checkResultForErrors
import ru.yandex.direct.utils.JsonUtils

open class GeminiClient(
    apiUrl: String,
    private val user: String,
    parallelFetcherFactory: ParallelFetcherFactory
) {
    companion object {
        private val logger: Logger = LoggerFactory.getLogger(GeminiClient::class.java)

        private const val RESPONSE_KEY = "Response"
    }

    private object CanonizationType {
        const val WEAK = "weak"
        const val MIRROR = "mirror"
    }

    private val api: GeminiApi = createApi(apiUrl, parallelFetcherFactory)

    private fun createApi(url: String, parallelFetcherFactory: ParallelFetcherFactory): GeminiApi {
        return Smart.builder()
            .withParallelFetcherFactory(parallelFetcherFactory)
            .withProfileName("gemini_client")
            .withBaseUrl(url)
            .build()
            .create(GeminiApi::class.java)
    }

    /**
     * Получить главные зеркала сайтов по переданным урлам
     */
    open fun getMainMirrors(urls: Collection<String>): Map<String, String> {
        val canonizedUrlsByOriginalUrl =
            executeAndProcess(urls, CanonizationType.MIRROR, GeminiSimpleTypeResponse::class.java)

        val result = HashMap<String, String>()
        for (url in urls) {
            val urlData = canonizedUrlsByOriginalUrl[url]
            if (urlData == null) {
                logger.error("Gemini response for url $url is not found")
                continue
            } else if (urlData.error != null) {
                logger.info("Url $url is not valid. Error: ${urlData.error}")
                continue
            }

            result[url] = urlData.canonizedUrl!!
        }

        return result
    }

    private fun <T> executeAndProcess(urls: Collection<String>, type: String,
                                      classOfResult: Class<T>): Map<String, T> {
        if (urls.isEmpty()) {
            return emptyMap()
        }

        val result = api.getCanonizedUrl(user, urls, type).execute()
        checkResultForErrors(result, ::GeminiClientException)

        val geminiResponse = result.success
        if (resultIsNotExists(geminiResponse)) {
            throw GeminiClientException("Can't process via gemini urls ($urls) " +
                "with canonization type $type - not found")
        }

        return getResponseValues(geminiResponse)
            .mapValues { JsonUtils.fromJson(it.value, classOfResult) }
    }

    private fun resultIsNotExists(response: GeminiResponse): Boolean {
        return response.totalDocCount.all { it <= 0 }
    }

    private fun getResponseValues(response: GeminiResponse): Map<String, String> {
        if (response.grouping == null) {
            return emptyMap()
        }

        return response.grouping.asSequence()
            .flatMap { it.group }
            .flatMap { it.document }
            .map { Pair(it.url, it.archiveInfo.gtaRelatedAttributes) }
            .flatMap { (url, attributes) ->
                val responseAttr = attributes.find { RESPONSE_KEY == it.key }
                if (responseAttr == null) {
                    logger.error("Can't process via gemini url ($url) - " +
                        "key \"$RESPONSE_KEY\" not found in gta-attributes\n$attributes")
                    emptySequence()
                } else {
                    sequenceOf(Pair(url, responseAttr.value))
                }
            }
            .associateBy(Pair<String, String>::first, Pair<String, String>::second)
    }

}
