package ru.yandex.direct.geobasehelper;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.direct.geobasehelper.exception.GeoBaseException;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.GeoTreeFactory;
import ru.yandex.geobase.CrimeaStatus;
import ru.yandex.geobase.Language;

import static java.util.stream.Collectors.toSet;

@ParametersAreNonnullByDefault
public abstract class GeoBaseHelper {
    static final Set<String> ALL_LANGUAGES = Arrays.stream(Language.values()).map(Enum::name).collect(toSet());

    private final GeoTreeFactory geoTreeFactory;

    GeoBaseHelper(GeoTreeFactory geoTreeFactory) {
        this.geoTreeFactory = geoTreeFactory;
    }

    /**
     * Некоторые регионы внутри Директа могут отсутствовать, поэтому если регион не найден,
     * пытаемся получить ближайшего региона-родителя, который поддерживается в директе
     */
    public Optional<Long> convertToDirectRegionId(@Nullable Long regionId) {
        return convertToDirectRegionId(regionId, null);
    }

    /**
     * То же, что и {@link GeoBaseHelper#convertToDirectRegionId(Long)} только смотрим на геодерево со стороны России
     * или Украины
     */
    public Optional<Long> convertToDirectRegionId(@Nullable Long regionId, @Nullable CrimeaStatus crimeaStatus) {
        if (regionId == null) {
            return Optional.empty();
        }

        if (getGeoTree().hasRegion(regionId)) {
            return Optional.of(regionId);
        }
        try {
            List<Integer> parentRegionIds = getParentRegionIds(regionId, crimeaStatus);
            return StreamEx.of(parentRegionIds)
                    .map(Integer::longValue)
                    .findFirst(getGeoTree()::hasRegion);
        } catch (GeoBaseException e) {
            return Optional.empty();
        }
    }

    /**
     * Находит наименьший регион, который есть в Директе, содержащий данную точку.
     */
    public Optional<Long> getDirectRegionId(double latitude, double longitude) {
        long externalRegion = getRegionIdByCoordinates(latitude, longitude);
        return convertToDirectRegionId(externalRegion, null);
    }

    /**
     * Поиск ближайшего коммерческого региона.
     * <p>
     * - Если регион найден в справочнике коммерческих регионов директа, тогда берем всех его детей и ищем регион
     * наиболее ближайший к точке, которая является входным параметром, проделываем это до тех пор, пока у
     * просматриваемого узла есть дети
     * - Если регион не найден, тогда поднимаемся вверх по иерархии регионов во внешнем сервисе http-geobase до тех пор,
     * пока регион не будет найден в справочнике коммерческих регионов директа
     * {@link GeoBaseHelper#convertToDirectRegionId(Long)}, а затем делаем спуск способом из предыдущего пункта
     *
     * @param crimeaStatus смотрим на дерево со стороны России или Украины
     * @return id региона в геобазе директа
     */
    public Optional<Long> getNearestDirectRegionId(double latitude, double longitude, CrimeaStatus crimeaStatus) {
        var externalRegion = getRegionIdByCoordinates(latitude, longitude);
        var foundRegion = convertToDirectRegionId(externalRegion, crimeaStatus);
        return foundRegion.flatMap(regionId -> getNearestDirectRegionId(latitude, longitude, regionId));
    }

    private Optional<Long> getNearestDirectRegionId(double latitude, double longitude, Long regionId) {
        Optional<Long> maybeRegionId;
        Optional<Long> maybePreviousRegionId = Optional.of(regionId);

        do {
            var minRegionDistancePair = getGeoTree()
                    .getChildren(maybePreviousRegionId.get())
                    .stream()
                    .map(childRegionId -> getDistanceRegionIdPair(latitude, longitude, childRegionId))
                    .min(Map.Entry.comparingByKey());

            maybeRegionId = minRegionDistancePair.map(Pair::getValue);

            if (minRegionDistancePair.isPresent()) {
                // Проверить исключительную ситуацию, когда родительский регион находится ближе, чем дети (потому что
                // дети могут быть, например, за МКАДом)
                var minDistance = minRegionDistancePair.map(Pair::getKey);
                var parentDistance = getDistanceRegionIdPair(latitude, longitude, maybePreviousRegionId.get()).getKey();
                if (minDistance.isPresent() && parentDistance < minDistance.get()) {
                    return maybePreviousRegionId;
                }
                maybePreviousRegionId = maybeRegionId;
            }
        } while (maybeRegionId.isPresent());

        return maybePreviousRegionId;
    }

    private Pair<Double, Long> getDistanceRegionIdPair(double latitude, double longitude, long regionId) {
        var coordinates = getCoordinatesByRegionId(regionId);
        var distance = getDistanceBetweenCoordinates(
                latitude,
                longitude,
                coordinates.getKey(),
                coordinates.getValue()
        );
        return Pair.of(distance, regionId);
    }

    /**
     * Отдает расстояние между точкой и коммерческим регионом в директе.
     */
    public abstract double getDistanceBetweenCoordinates(
            double latitude0,
            double longitude0,
            double latitude1,
            double longitude1
    );

    /**
     * Обратный геокодинг.
     */
    public abstract long getRegionIdByCoordinates(double latitude, double longitude);

    /**
     * Отдает регионы родителей в порядке от ближайшего к дальнейшему
     */
    public abstract List<Integer> getParentRegionIds(Long regionId);

    /**
     * То же, что и {@link GeoBaseHelper#getParentRegionIds(Long)}, только смотрим со стороны России или Украины
     */
    public abstract List<Integer> getParentRegionIds(Long regionId, @Nullable CrimeaStatus crimeaStatus);

    /**
     * Отдает имя региона по id
     */
    public abstract String getRegionName(Long regionId);

    /**
     * Отдает имя региона по id на заданном языке.
     *
     * @param regionId Идентификатор региона в Геобазе.
     * @param lang     Двубуквенный код языка. Примеры: RU, TR, UA.
     */
    public abstract String getRegionName(Long regionId, String lang);

    /**
     * Отдает id страны, к которой относится данный регион
     */
    public abstract int getCountryId(Long regionId);

    /**
     * Отдает телефонные коды региона
     */
    public abstract String getPhoneCodeByRegionId(Long regionId);

    /**
     * Отдает координаты
     */
    public abstract Pair<Double, Double> getCoordinatesByRegionId(long regionId);

    public abstract long getRegionIdByIp(String ip);

    /**
     * Возвращает имя часового пояса региона, например "Europe/Moscow"
     */
    public abstract String getTimezoneByRegionId(Long regionId);

    /**
     * Возвращает идентификатор региона, обозначенного как главный для указанного региона
     */
    public abstract long getChiefRegionId(Long regionId);

    private GeoTree getGeoTree() {
        return geoTreeFactory.getGlobalGeoTree();
    }
}
