package ru.yandex.direct.geobasehelper;

import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.ws.rs.ProcessingException;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.client.ResponseProcessingException;

import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.geobasehelper.exception.GeoBaseException;
import ru.yandex.direct.regions.GeoTreeFactory;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.geobase.CrimeaStatus;
import ru.yandex.geobase.HttpGeobase;
import ru.yandex.geobase.beans.GeobaseRegionData;
import ru.yandex.geobase.beans.GeobaseRegionTranslations;

import static com.google.common.base.Preconditions.checkArgument;

/**
 * Реализация работы с геобазой, работающая с http-апи геобазы.
 */
@ParametersAreNonnullByDefault
public class GeoBaseHttpApiHelper extends GeoBaseHelper {
    private static final Logger logger = LoggerFactory.getLogger(GeoBaseHttpApiHelper.class);

    private final HttpGeobase httpGeoBase;

    public GeoBaseHttpApiHelper(GeoTreeFactory geoTreeFactory) {
        super(geoTreeFactory);
        this.httpGeoBase = HttpGeobase.getInstance();
    }

    @Override
    public double getDistanceBetweenCoordinates(
            double latitude0,
            double longitude0,
            double latitude1,
            double longitude1
    ) {
        logger.trace("get distance between coordinates ({}, {}) and ({}, {})", latitude0, longitude0, latitude1, latitude1);
        //noinspection unused
        try (TraceProfile profile = Trace.current().profile("geobase:getRegionId", "", 1)) {
            return httpGeoBase.getDistance(latitude0, longitude0, latitude1, longitude1);
        } catch (ResponseProcessingException e) {
            logger.error(
                    "Processing of a received HTTP response fails for coordinates: ({}, {}); ({}, {})",
                    latitude0,
                    longitude0,
                    latitude1,
                    longitude1
            );
            throw new GeoBaseException(e);
        } catch (ProcessingException e) {
            logger.error(
                    "Processing of request fails for coordinates: ({}, {}); ({}, {})",
                    latitude0,
                    longitude0,
                    latitude1,
                    longitude1
            );
            throw new GeoBaseException(e);
        } catch (WebApplicationException e) {
            logger.error(
                    "Status code of the response is not successful for coordinates: ({}, {}); ({}, {})",
                    latitude0,
                    longitude0,
                    latitude1,
                    longitude1
            );
            throw new GeoBaseException(e);
        }
    }

    @Override
    public long getRegionIdByCoordinates(double latitude, double longitude) {
        logger.trace("get region ID by coordinates: {}, {}", latitude, longitude);
        //noinspection unused
        try (TraceProfile profile = Trace.current().profile("geobase:getRegionId", "", 1)) {
            return httpGeoBase.getRegionId(latitude, longitude);
        } catch (ResponseProcessingException e) {
            logger.error("Processing of a received HTTP response fails for coordinates: {}, {}", latitude, longitude);
            throw new GeoBaseException(e);
        } catch (ProcessingException e) {
            logger.error("Processing of request fails for coordinates: {}, {}", latitude, longitude);
            throw new GeoBaseException(e);
        } catch (WebApplicationException e) {
            logger.error("Status code of the response is not successful for coordinates: {}, {}", latitude, longitude);
            throw new GeoBaseException(e);
        }
    }

    @Override
    public List<Integer> getParentRegionIds(Long regionId) throws GeoBaseException {
        return getParentRegionIds(regionId, null);
    }

    @Override
    public List<Integer> getParentRegionIds(Long regionId, @Nullable CrimeaStatus crimeaStatus) throws GeoBaseException {
        logger.trace("get parent region ids for regionId: {}", regionId);
        //noinspection unused
        try (TraceProfile profile = Trace.current().profile("geobase:getParents", "", 1)) {
            return httpGeoBase.getParents(regionId.intValue(), crimeaStatus);
        } catch (ResponseProcessingException e) {
            logger.error("Processing of a received HTTP response fails for regionId: {}", regionId);
            throw new GeoBaseException(e);
        } catch (ProcessingException e) {
            logger.error("Processing of request fails for regionId: {}", regionId);
            throw new GeoBaseException(e);
        } catch (WebApplicationException e) {
            logger.error("Status code of the response is not successful for regionId: {}", regionId);
            throw new GeoBaseException(e);
        }
    }

    @Override
    public String getRegionName(Long regionId, String lang) {
        checkArgument(ALL_LANGUAGES.contains(lang), "Language '%s' is unknown.", lang);
        logger.trace("get translations for regionId: {}, lang {}", regionId, lang);
        //noinspection unused
        try (TraceProfile profile = Trace.current().profile("geobase:getTranslations", "", 1)) {
            GeobaseRegionTranslations translations =
                    httpGeoBase.getTranslations(regionId.intValue(), lang.toUpperCase());
            return translations.getNominativeCase();
        } catch (ResponseProcessingException e) {
            logger.error("Processing of a received HTTP response fails for regionId: {}, lang {}", regionId, lang);
            throw new GeoBaseException(e);
        } catch (ProcessingException e) {
            logger.error("Processing of request fails for regionId: {}, lang {}", regionId, lang);
            throw new GeoBaseException(e);
        } catch (WebApplicationException e) {
            logger.error("Status code of the response is not successful for regionId: {}, lang {}", regionId, lang);
            throw new GeoBaseException(e);
        }
    }

    @Override
    public int getCountryId(Long regionId) throws GeoBaseException {
        logger.trace("get country for regionId: {}", regionId);
        //noinspection unused
        try (TraceProfile profile = Trace.current().profile("geobase:getCountryId", "", 1)) {
            return httpGeoBase.getCountryId(regionId.intValue());
        } catch (ResponseProcessingException e) {
            logger.error("Processing of a received HTTP response fails for regionId: {}", regionId);
            throw new GeoBaseException(e);
        } catch (ProcessingException e) {
            logger.error("Processing of request fails for regionId: {}", regionId);
            throw new GeoBaseException(e);
        } catch (WebApplicationException e) {
            logger.error("Status code of the response is not successful for regionId: {}", regionId);
            throw new GeoBaseException(e);
        }
    }

    @Override
    public String getRegionName(Long regionId) throws GeoBaseException {
        logger.trace("get region by id: {}", regionId);
        return getRegionById(regionId, "getRegion").getName();
    }

    @Override
    public String getPhoneCodeByRegionId(Long regionId) throws GeoBaseException {
        logger.trace("get phone code for regionId: {}", regionId);
        return getRegionById(regionId, "phoneCode").getPhoneCode();
    }

    @Override
    public Pair<Double, Double> getCoordinatesByRegionId(long regionId) {
        logger.trace("get coordinates for regionId: {}", regionId);
        var region = getRegionById(regionId, "coordinates");
        return Pair.of(region.getLatitude(), region.getLongitude());
    }

    @Override
    public long getRegionIdByIp(String ip) {
        logger.trace("get region id for ip: {}", ip);
        return httpGeoBase.getRegionId(ip);
    }

    @Override
    public String getTimezoneByRegionId(Long regionId) {
        logger.trace("get timezone for regionId: {}", regionId);
        try (TraceProfile profile = Trace.current().profile("geobase:getTimezone", "", 1)) {
            return httpGeoBase.getTimezone(regionId.intValue());
        } catch (ResponseProcessingException e) {
            logger.error("Processing of a received HTTP response fails for regionId: {}", regionId);
            throw new GeoBaseException(e);
        } catch (ProcessingException e) {
            logger.error("Processing of request fails for regionId: {}", regionId);
            throw new GeoBaseException(e);
        } catch (WebApplicationException e) {
            logger.error("Status code of the response is not successful for regionId: {}", regionId);
            throw new GeoBaseException(e);
        }
    }

    @Override
    public long getChiefRegionId(Long regionId) {
        logger.trace("get chief region for regionId: {}", regionId);
        try (TraceProfile profile = Trace.current().profile("geobase:getChiefRegionId", "", 1)) {
            return httpGeoBase.getChiefRegionId(regionId.intValue());
        } catch (ResponseProcessingException e) {
            logger.error("Processing of a received HTTP response fails for regionId: {}", regionId);
            throw new GeoBaseException(e);
        } catch (ProcessingException e) {
            logger.error("Processing of request fails for regionId: {}", regionId);
            throw new GeoBaseException(e);
        } catch (WebApplicationException e) {
            logger.error("Status code of the response is not successful for regionId: {}", regionId);
            throw new GeoBaseException(e);
        }
    }

    //noinspection unused
    private GeobaseRegionData getRegionById(Long regionId, String traceSuffix) {
        try (TraceProfile profile = Trace.current().profile("geobase:" + traceSuffix, "", 1)) {
            return httpGeoBase.getRegion(regionId.intValue());
        } catch (ResponseProcessingException e) {
            logger.error("Processing of a received HTTP response fails for regionId: {}", regionId);
            throw new GeoBaseException(e);
        } catch (ProcessingException e) {
            logger.error("Processing of request fails for regionId: {}", regionId);
            throw new GeoBaseException(e);
        } catch (WebApplicationException e) {
            logger.error("Status code of the response is not successful for regionId: {}", regionId);
            throw new GeoBaseException(e);
        }
    }
}
