package ru.yandex.direct.geosearch;

import java.util.Collection;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import yandex.maps.proto.common2.geo_object.GeoObjectOuterClass;
import yandex.maps.proto.common2.metadata.MetadataOuterClass;
import yandex.maps.proto.common2.response.ResponseOuterClass;
import yandex.maps.proto.search.business.Business;
import yandex.maps.proto.search.experimental.Experimental;

import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.geosearch.model.Address;
import ru.yandex.direct.geosearch.model.AddressComponent;
import ru.yandex.direct.geosearch.model.GeoObject;
import ru.yandex.direct.geosearch.model.Kind;
import ru.yandex.direct.geosearch.model.Lang;
import ru.yandex.direct.geosearch.model.Precision;
import ru.yandex.direct.http.smart.converter.ResponseConverterFactory;
import ru.yandex.direct.http.smart.core.Smart;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;

import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static ru.yandex.direct.geosearch.Api.CLUSTER_PERMALINKS;
import static ru.yandex.direct.http.smart.error.ErrorUtils.checkResultForErrors;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class GeosearchClient {

    static final EnumSet<Kind> ALLOWABLE_KINDS = EnumSet.of(
            Kind.HOUSE, Kind.STREET, Kind.LOCALITY, Kind.METRO, Kind.KM, Kind.RAILWAY, Kind.OTHER);
    static final GeoObject DEFAULT_GEO_OBJECT = new GeoObject.Builder()
            .withX(37.611643).withY(55.819692)
            .withX1(11.066975).withY1(43.267788)
            .withX2(177.356039).withY2(74.690828)
            .withPrecision(Precision.OTHER)
            .withKind(Kind.UNKNOWN)
            .build();

    private final Api api;
    private final GeosearchClientSettings settings;

    public GeosearchClient(GeosearchClientSettings geosearchSettings, TvmService tvmService,
                           ParallelFetcherFactory parallelFetcherFactory,
                           TvmIntegration tvmIntegration) {
        settings = geosearchSettings;
        api = createApi(settings.getApiUrl(), tvmIntegration, tvmService, parallelFetcherFactory);
    }

    private Api createApi(String url, TvmIntegration tvmIntegration, TvmService tvmService,
                          ParallelFetcherFactory parallelFetcherFactory) {
        return Smart.builder()
                .withParallelFetcherFactory(parallelFetcherFactory)
                .withResponseConverterFactory(ResponseConverterFactory.builder()
                        .addConverters(new GeosearchResponseConverter())
                        .build())
                .withProfileName("geosearch_client")
                .useTvm(tvmIntegration, tvmService)
                .withBaseUrl(url)
                .build()
                .create(Api.class);
    }

    /**
     * Запрос к API Яндекс.Карт с преобразованием protobuf объектов к списку описанных моделей адреса
     * <p>
     * В случае ошибки парсинга ответа будет выброшено исключение
     **/
    public List<GeoObject> searchAddress(String text, Lang lang) {
        Result<ResponseOuterClass.Response> response = api.searchAddress(
                text, lang.getLocale(), settings.getOrigin()
        ).execute();
        checkResultForErrors(response, GeosearchClientException::new);

        try {
            return mapList(response.getSuccess().getReply().getGeoObjectList(), GeoObject::new);
        } catch (RuntimeException e) {
            throw new GeosearchClientException("Error in parsing response and building geosearch answer models." + e);
        }
    }

    public List<GeoObject> searchAddress(String text) {
        return searchAddress(text, Lang.RU);
    }

    public List<GeoObject> searchAddress(Address address, Lang lang) {
        return searchAddress(address.toString(), lang);
    }

    public List<GeoObject> searchAddress(Address address) {
        return searchAddress(address.toString(), Lang.RU);
    }

    public List<GeoObject> searchMetro(String text, Lang lang) {
        Result<ResponseOuterClass.Response> response = api.searchMetro(
                text, lang.getLocale(), settings.getOrigin()
        ).execute();
        return fetchGeoObjects(response);
    }

    public List<GeoObject> searchReverse(String coordinates, Kind kind, Lang lang) {
        String kindParameter = ifNotNull(kind, Kind::getQueryParameterValue);
        Result<ResponseOuterClass.Response> response = api.searchReverse(
                coordinates, kindParameter, lang.getLocale(), settings.getOrigin()
        ).execute();
        return fetchGeoObjects(response);
    }

    private List<GeoObject> fetchGeoObjects(Result<ResponseOuterClass.Response> searchResponse) {
        checkResultForErrors(searchResponse, GeosearchClientException::new);
        try {
            return mapList(searchResponse.getSuccess().getReply().getGeoObjectList(), GeoObject::new);
        } catch (RuntimeException e) {
            throw new GeosearchClientException("Error in parsing response and building geosearch answer models." + e);
        }
    }

    @Nonnull
    private static String normalizeCity(String city) {
        // Может не совпадать в запросе и ответе геокодера. Пример: запрос - Орёл, ответ - Орел (См. CommonMaps
        // .pm::_valid_point)
        return StringUtils.replaceChars(city, "ёЁ", "еЕ");
    }

    /**
     * Возвращает пару город, административная единица в которой находится город (см. CommonMaps
     * .pm::_select_first_point)
     */
    private static Pair<String, String> splitCityToCityAndAdministrativeArea(@Nullable String city) {
        if (city == null) {
            return Pair.of("", "");
        }

        // Как в perl-е ищем первое вхождение pattern-а \((.*?)\)
        int firstLParent = city.indexOf('(');
        if (firstLParent == -1) {
            return Pair.of(city, "");
        }

        int firstRParent = city.indexOf(')', firstLParent + 1);
        if (firstRParent == -1) {
            return Pair.of(city, "");
        }

        // Находим первый непробельный символ с конца перед '('
        int endOfCity = firstLParent;
        while (endOfCity > 0 && Character.isWhitespace(city.charAt(endOfCity - 1))) {
            endOfCity--;
        }

        return Pair.of(
                city.substring(0, endOfCity), city.substring(firstLParent + 1, firstRParent));
    }


    private static boolean isCityMatches(String normalizedExpected, @Nullable String actual) {
        //noinspection SimplifiableIfStatement
        if (StringUtils.isEmpty(actual)) {
            return false;
        }
        return normalizedExpected.equalsIgnoreCase(normalizeCity(actual));
    }

    private static boolean isAdministrativeAreaMatches(String expected, @Nullable String actual) {
        //noinspection SimplifiableIfStatement
        if (StringUtils.isEmpty(actual)) {
            // В perl-е сравниваются только если присутствует в ответе (См. CommonMaps.pm::_valid_point)
            return true;
        }
        return expected.equalsIgnoreCase(actual);
    }

    /**
     * Пытаемся найти точку наиболее соответствующую данному адресу (См. CommonMaps.pm::check_address_map)
     * TODO: Надо вынести метод в ядро (https://st.yandex-team.ru/DIRECT-70828)
     */
    public GeoObject getMostRelevantGeoData(Address address) {
        Pair<String, String> cityAndAdministrativeArea = splitCityToCityAndAdministrativeArea(address.getCity());

        String normalizedCity = normalizeCity(cityAndAdministrativeArea.getLeft());
        String administrativeArea = cityAndAdministrativeArea.getRight();

        List<GeoObject> geoData = searchAddress(address);

        Stream<GeoObject> geoObjects = geoData
                .stream()
                .filter(go -> ALLOWABLE_KINDS.contains(go.getKind()));

        if (!StringUtils.isEmpty(normalizedCity)) {
            geoObjects = geoObjects.filter(
                    go -> isCityMatches(normalizedCity, go.getCity())
                            || isCityMatches(normalizedCity, go.getAdministrativeArea()));
        }

        if (!StringUtils.isEmpty(administrativeArea)) {
            geoObjects = geoObjects.filter(
                    go -> isAdministrativeAreaMatches(administrativeArea, go.getAdministrativeArea()));
        }

        return geoObjects.min(Comparator.comparing(GeoObject::getPrecision))
                // Если ничего не найдено то зуммим на всю Россию и точку ставим в Москву
                // (См. CommonMaps.pm::_select_first_point)
                .orElse(DEFAULT_GEO_OBJECT);
    }

    /**
     * По координатам определяет самый точный адрес указанного типа.
     * <p>
     * Адрес состоит из последовательности компонентов разных типов,
     * и тип адреса определяется типом последнего компонента. Например,
     * "Россия, Москва" - это адрес типа "province", состоящий из компонентов
     * типа "country" и "province".
     * <p>
     * При поиске по координатам геокодер для каждого самого точного найденного адреса всегда возвращает
     * все более общие адреса, входящие в него. Например, для Москвы всегда вернется два адреса:
     * "Россия, Москва" и "Россия". Адреса, входящие друг в друга далее будем называть адресами, принадлежащими
     * одной иерархии. В противоположность, "Россия, Московская область" и "Россия, Москва" - адреса
     * из разных иерархий, хоть и пересекающихся на уровне страны.
     * <p>
     * В зависимости от того, какой адрес или координату передать, геокодер может вернуть
     * одну или несколько иерархий адресов. Например, одновременно: "Россия, Северо-Западный федеральный округ,
     * Санкт-Петербург, Курортный район", "Россия, Северо-Западный федеральный округ, Санкт-Петербург"
     * (из той же иерархии, но более общий) и "Россия, Северо-Западный федеральный округ, Ленинградская область" -
     * из другой иерархии. При этом и Ленинградская область и Санкт-Петербург имеют один и тот же тип - "province",
     * и одинаковое количество компонентов (глубину).
     * <p>
     * Так же в одном адресе могут находиться несколько компонентов одного типа. В примерах выше и
     * "Северо-Западный федеральный округ", и "Санкт-Петербург", и "Ленинградская область" имеют один
     * и тот же тип - "province".
     * <p>
     * Поэтому, алгоритм поиска наиболее точного адреса указанного типа такой.
     * Находим самый длинный адрес (состоящий из наибольшего числа компонентов), который включает в себя
     * компонент искомого типа, и считаем, что это самый точный адрес с данным компонентом.
     * Далее находим самый длинный адрес указанного типа, с которого начинается найденный на предыдущем шаге адрес.
     * Это и будет искомый адрес.
     * <p>
     * Если не очень понятно, можно посмотреть тест-кейсы.
     * <p>
     * Если в указанном адресе отсутствуют компоненты заданного типа, то вернет пустой Optional.
     *
     * @param coordinates координаты точки, формат описан здесь:
     *                    https://tech.yandex
     *                    .ru/maps/doc/geocoder/desc/concepts/input_params-docpage/#input_params__geocode-format
     * @param kind        тип адреса, который необходимо получить.
     */
    public Optional<GeoObject> getMostExactGeoObjectOfKind(String coordinates, Kind kind) {
        List<GeoObject> geoObjects = searchAddress(coordinates);

        Predicate<GeoObject> componentsContainKindPredicate =
                geoObject -> StreamEx.of(geoObject.getComponents())
                        .map(AddressComponent::getKind)
                        .toSet()
                        .contains(kind);

        // все адреса, содержащие компонент указанного типа
        // могут принадлежать разным иерархиям адресов
        List<GeoObject> geoObjectsContainingKind = StreamEx.of(geoObjects)
                .filter(geoObject -> geoObject.getComponents() != null)
                .filter(componentsContainKindPredicate)
                .toList();

        // самый длинный адрес, содержащий компонент указанного типа
        Optional<GeoObject> maxDepthGeoObjectContainingKind = StreamEx.of(geoObjectsContainingKind)
                .maxBy(geoObject -> geoObject.getComponents().size());

        if (maxDepthGeoObjectContainingKind.isEmpty()) {
            return Optional.empty();
        }

        // компоненты самого длинного адреса
        List<AddressComponent> maxDepthComponents = maxDepthGeoObjectContainingKind.get().getComponents();

        // определяет, начинается ли самый длинный адрес с переданного адреса
        Predicate<GeoObject> geoObjectIsStartOfMaxDepthGeoObject = geoObject -> {
            List<AddressComponent> curObjectComponents = geoObject.getComponents();
            if (curObjectComponents.size() > maxDepthComponents.size()) {
                return false;
            }
            for (int i = 0; i < curObjectComponents.size(); i++) {
                if (!curObjectComponents.get(i).equals(maxDepthComponents.get(i))) {
                    return false;
                }
            }
            return true;
        };

        // самый длинный адрес указанного типа, с компонентов которого начинается
        // самый длинный адрес, содержащий компонент указанного типа
        return StreamEx.of(geoObjectsContainingKind)
                .filter(geoObject -> kind.equals(geoObject.getKind()))
                .filter(geoObjectIsStartOfMaxDepthGeoObject)
                .maxBy(geoObject -> geoObject.getComponents().size());
    }

    /**
     * Возвращает склеенные пермалинки, сгруппированные по их общиему головному пермалинку.
     *
     * @param permalinks Список пермалинков
     */
    public Map<Long, Set<Long>> getMergedPermalinks(Collection<Long> permalinks) {
        if (permalinks.isEmpty()) {
            return emptyMap();
        }

        Result<ResponseOuterClass.Response> apiResponse = api
                .searchOldPermalinks(settings.getOrigin(), mapList(permalinks, String::valueOf))
                .execute();
        checkResultForErrors(apiResponse, GeosearchClientException::new);
        ResponseOuterClass.Response response = apiResponse.getSuccess();

        Map<Long, Set<Long>> result = new HashMap<>();
        for (GeoObjectOuterClass.GeoObject geoObject : response.getReply().getGeoObjectList()) {
            Optional<Long> headPermalinkId = StreamEx.of(geoObject.getMetadataList())
                    .findFirst(metadata -> metadata.hasExtension(Business.gEOOBJECTMETADATA))
                    .map(metadata -> metadata.getExtension(Business.gEOOBJECTMETADATA))
                    .map(Business.GeoObjectMetadata::getId)
                    .map(Long::valueOf);
            if (headPermalinkId.isPresent()) {
                Set<Long> clusterPermalinksFromGeosearch = getObjectClusterPermalinks(geoObject);
                Set<Long> clusterPermalinks = clusterPermalinksFromGeosearch.isEmpty()
                        ? Set.of(headPermalinkId.get())
                        : clusterPermalinksFromGeosearch;
                result.put(headPermalinkId.get(), clusterPermalinks);
            }
        }
        return result;
    }

    /**
     * Для объекта {@link yandex.maps.proto.common2.geo_object.GeoObjectOuterClass.GeoObject} вытаскивает из меты
     * поле {@link Api#CLUSTER_PERMALINKS} и делает из него {@code Set<Long>}.
     * Возвращает пустой сет если такого поля нет либо если оно пустое.
     */
    private Set<Long> getObjectClusterPermalinks(GeoObjectOuterClass.GeoObject geoObject) {
        for (MetadataOuterClass.Metadata metadata : geoObject.getMetadataList()) {
            if (!metadata.hasExtension(Experimental.gEOOBJECTMETADATA)) {
                continue;
            }

            Optional<Experimental.ExperimentalStorage.Item> clusterPermalinks = metadata
                    .getExtension(Experimental.gEOOBJECTMETADATA)
                    .getExperimentalStorage()
                    .getItemList()
                    .stream()
                    .filter(item -> CLUSTER_PERMALINKS.equals(item.getKey()))
                    .findFirst();

            if (clusterPermalinks.isEmpty()) {
                return emptySet();
            }

            return StreamEx
                    .of(clusterPermalinks.get().getValue().split(","))
                    .map(Long::valueOf)
                    .toSet();
        }
        return emptySet();
    }
}
