package ru.yandex.direct.geosearch.model;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.List;

import com.google.common.collect.ImmutableList;
import org.apache.commons.lang3.EnumUtils;
import yandex.maps.proto.common2.geo_object.GeoObjectOuterClass;
import yandex.maps.proto.common2.geometry.GeometryOuterClass;
import yandex.maps.proto.search.address.AddressOuterClass;
import yandex.maps.proto.search.geocoder.Geocoder;
import yandex.maps.proto.search.geocoder_internal.GeocoderInternal;
import yandex.maps.proto.search.kind.KindOuterClass;
import yandex.maps.proto.search.precision.PrecisionOuterClass;

import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

public class GeoObject {

    private final Long geoId;
    private final Long houses;
    private String country;
    private String administrativeArea;
    private String city;
    private String street;
    private String house;
    private String text;
    private String name;

    private final BigDecimal x;
    private final BigDecimal y;
    private final BigDecimal x1;
    private final BigDecimal y1;
    private final BigDecimal x2;
    private final BigDecimal y2;

    private final Precision precision;
    private final Kind kind;

    private final ImmutableList<AddressComponent> components;

    private GeoObject(
            Long geoId,
            Long houses,
            String country,
            String administrativeArea,
            String city,
            String street,
            String house,
            String text,
            String name,
            BigDecimal x, BigDecimal y,
            BigDecimal x1, BigDecimal y1, BigDecimal x2, BigDecimal y2,
            Precision precision,
            Kind kind,
            List<AddressComponent> components) {
        this.geoId = geoId;
        this.houses = houses;
        this.country = country;
        this.administrativeArea = administrativeArea;
        this.city = city;
        this.street = street;
        this.house = house;
        this.text = text;
        this.name = name;
        this.x = x;
        this.y = y;
        this.x1 = x1;
        this.y1 = y1;
        this.x2 = x2;
        this.y2 = y2;
        this.precision = precision;
        this.kind = kind;
        this.components = ifNotNull(components, ImmutableList::copyOf);
    }

    /**
     * Собирает модель из protobuf объекта
     */
    public GeoObject(GeoObjectOuterClass.GeoObject responseGeoObject) {

        Geocoder.GeoObjectMetadata gcMetadata = getGeoObjectMetadata(responseGeoObject);
        geoId = (long) gcMetadata.getExtension(GeocoderInternal.tOPONYMINFO).getGeoid();
        houses = (long) gcMetadata.getExtension(GeocoderInternal.tOPONYMINFO).getHouseCount();

        AddressOuterClass.Address address = gcMetadata.getAddress();
        List<AddressOuterClass.Component> responseComponents = address.getComponentList();

        List<AddressComponent> addressComponents = new ArrayList<>();

        for (AddressOuterClass.Component component:responseComponents) {
            KindOuterClass.Kind kind = component.getKindList().get(0);
            if (kind.equals(KindOuterClass.Kind.COUNTRY)){
                country = component.getName();
            } else if (kind.equals(KindOuterClass.Kind.AREA)){
                administrativeArea = component.getName();
            } else if (kind.equals(KindOuterClass.Kind.LOCALITY)){
                city = component.getName();
            } else if (kind.equals(KindOuterClass.Kind.STREET)){
                street = component.getName();
            } else if (kind.equals(KindOuterClass.Kind.HOUSE)){
                house = component.getName();
            } else if (kind.equals(KindOuterClass.Kind.ROUTE)){
                if (street == null){
                    street = component.getName();
                }
            }

            if (kind.equals(KindOuterClass.Kind.METRO_STATION)){
                addressComponents.add(new AddressComponent(Kind.METRO, component.getName()));
            } else if (kind.equals(KindOuterClass.Kind.RAILWAY_STATION)) {
                addressComponents.add(new AddressComponent(Kind.RAILWAY, component.getName()));
            } else if (!EnumUtils.isValidEnum(Kind.class, kind.toString())) {
                addressComponents.add(new AddressComponent(Kind.OTHER, component.getName()));
            } else {
                addressComponents.add(new AddressComponent(Kind.valueOf(kind.toString()), component.getName()));
            }
        }
        components = ImmutableList.copyOf(addressComponents);
        kind = Kind.valueOf(addressComponents.get(addressComponents.size() - 1).getKind().toString());
        if (gcMetadata.getHousePrecision().equals(PrecisionOuterClass.Precision.NEARBY)) {
            precision = Precision.NEAR;
        } else if (gcMetadata.getHousePrecision().equals(PrecisionOuterClass.Precision.RANGE)) {
            precision = Precision.NEAR;
        } else {
            precision = Precision.valueOf(gcMetadata.getHousePrecision().toString());
        }
        text = address.getFormattedAddress();
        name = responseGeoObject.getName();

        GeometryOuterClass.BoundingBox boundedBy = responseGeoObject.getBoundedBy();
        GeometryOuterClass.Point lowerCorner = boundedBy.getLowerCorner();
        x1 = new BigDecimal(lowerCorner.getLon()).setScale(6, RoundingMode.HALF_UP).stripTrailingZeros();
        y1 = new BigDecimal(lowerCorner.getLat()).setScale(6, RoundingMode.HALF_UP).stripTrailingZeros();

        GeometryOuterClass.Point upperCorner = boundedBy.getUpperCorner();
        x2 = new BigDecimal(upperCorner.getLon()).setScale(6, RoundingMode.HALF_UP).stripTrailingZeros();
        y2 = new BigDecimal(upperCorner.getLat()).setScale(6, RoundingMode.HALF_UP).stripTrailingZeros();

        GeometryOuterClass.Point point = responseGeoObject.getGeometryList().get(0).getPoint();
        x = new BigDecimal(point.getLon()).setScale(6, RoundingMode.HALF_UP).stripTrailingZeros();
        y = new BigDecimal(point.getLat()).setScale(6, RoundingMode.HALF_UP).stripTrailingZeros();

    }

    public Long getGeoId() {
        return geoId;
    }

    public Long getHouses() {
        return houses;
    }

    public String getCountry() {
        return country;
    }

    public String getAdministrativeArea() {
        return administrativeArea;
    }

    public String getCity() {
        return city;
    }

    public String getStreet() {
        return street;
    }

    public String getHouse() {
        return house;
    }

    public String getText() {
        return text;
    }

    public String getName() {
        return name;
    }

    public BigDecimal getX() {
        return x;
    }

    public BigDecimal getY() {
        return y;
    }

    public BigDecimal getX1() {
        return x1;
    }

    public BigDecimal getY1() {
        return y1;
    }

    public BigDecimal getX2() {
        return x2;
    }

    public BigDecimal getY2() {
        return y2;
    }

    public Precision getPrecision() {
        return precision;
    }

    public Kind getKind() {
        return kind;
    }

    public ImmutableList<AddressComponent> getComponents() {
        return components;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        GeoObject geoObject = (GeoObject) o;

        if (geoId != null ? !geoId.equals(geoObject.geoId) : geoObject.geoId != null) {
            return false;
        }
        if (houses != null ? !houses.equals(geoObject.houses) : geoObject.houses != null) {
            return false;
        }
        if (country != null ? !country.equals(geoObject.country) : geoObject.country != null) {
            return false;
        }
        if (administrativeArea != null ? !administrativeArea.equals(geoObject.administrativeArea)
                : geoObject.administrativeArea != null) {
            return false;
        }
        if (city != null ? !city.equals(geoObject.city) : geoObject.city != null) {
            return false;
        }
        if (street != null ? !street.equals(geoObject.street) : geoObject.street != null) {
            return false;
        }
        if (house != null ? !house.equals(geoObject.house) : geoObject.house != null) {
            return false;
        }
        if (text != null ? !text.equals(geoObject.text) : geoObject.text != null) {
            return false;
        }
        if (name != null ? !name.equals(geoObject.name) : geoObject.name != null) {
            return false;
        }
        if (x != null ? !x.equals(geoObject.x) : geoObject.x != null) {
            return false;
        }
        if (y != null ? !y.equals(geoObject.y) : geoObject.y != null) {
            return false;
        }
        if (x1 != null ? !x1.equals(geoObject.x1) : geoObject.x1 != null) {
            return false;
        }
        if (y1 != null ? !y1.equals(geoObject.y1) : geoObject.y1 != null) {
            return false;
        }
        if (x2 != null ? !x2.equals(geoObject.x2) : geoObject.x2 != null) {
            return false;
        }
        if (y2 != null ? !y2.equals(geoObject.y2) : geoObject.y2 != null) {
            return false;
        }
        if (precision != geoObject.precision) {
            return false;
        }
        if (kind != geoObject.kind) {
            return false;
        }
        return components != null ? components.equals(geoObject.components) : geoObject.components == null;
    }

    @Override
    public int hashCode() {
        int result = geoId != null ? geoId.hashCode() : 0;
        result = 31 * result + (houses != null ? houses.intValue() : 0);
        result = 31 * result + (country != null ? country.hashCode() : 0);
        result = 31 * result + (administrativeArea != null ? administrativeArea.hashCode() : 0);
        result = 31 * result + (city != null ? city.hashCode() : 0);
        result = 31 * result + (street != null ? street.hashCode() : 0);
        result = 31 * result + (house != null ? house.hashCode() : 0);
        result = 31 * result + (text != null ? text.hashCode() : 0);
        result = 31 * result + (name != null ? name.hashCode() : 0);
        result = 31 * result + (x != null ? x.hashCode() : 0);
        result = 31 * result + (y != null ? y.hashCode() : 0);
        result = 31 * result + (x1 != null ? x1.hashCode() : 0);
        result = 31 * result + (y1 != null ? y1.hashCode() : 0);
        result = 31 * result + (x2 != null ? x2.hashCode() : 0);
        result = 31 * result + (y2 != null ? y2.hashCode() : 0);
        result = 31 * result + (precision != null ? precision.hashCode() : 0);
        result = 31 * result + (kind != null ? kind.hashCode() : 0);
        result = 31 * result + (components != null ? components.hashCode() : 0);
        return result;
    }

    @Override
    public String toString() {
        return "GeoObject{" +
                "geoId=" + geoId +
                ", houses='" + houses + '\'' +
                ", country='" + country + '\'' +
                ", administrativeArea='" + administrativeArea + '\'' +
                ", city='" + city + '\'' +
                ", street='" + street + '\'' +
                ", house='" + house + '\'' +
                ", text='" + text + '\'' +
                ", name='" + name + '\'' +
                ", x=" + x +
                ", y=" + y +
                ", x1=" + x1 +
                ", y1=" + y1 +
                ", x2=" + x2 +
                ", y2=" + y2 +
                ", precision=" + precision +
                ", kind=" + kind +
                ", components=" + components +
                '}';
    }

    public static Geocoder.GeoObjectMetadata getGeoObjectMetadata(GeoObjectOuterClass.GeoObject geoObject) {
        return geoObject.getMetadataList().stream()
                .filter(md -> md.hasExtension(Geocoder.gEOOBJECTMETADATA))
                .findAny().orElseThrow(() -> new RuntimeException("No Response GeoObjectMetadata"))
                .getExtension(Geocoder.gEOOBJECTMETADATA);
    }

    public static final class Builder {
        private Long geoId;
        private Long houses;

        private String country;
        private String administrativeArea;
        private String city;
        private String street;
        private String house;
        private String text;
        private String name;

        private BigDecimal x;
        private BigDecimal y;
        private BigDecimal x1;
        private BigDecimal y1;
        private BigDecimal x2;
        private BigDecimal y2;

        private Precision precision;
        private Kind kind;

        private List<AddressComponent> components;

        public Long getGeoId() {
            return geoId;
        }

        public Builder withGeoId(Long geoId) {
            this.geoId = geoId;
            return this;
        }

        public Long getHouses() {
            return houses;
        }

        public Builder withHouses(Long houses) {
            this.houses = houses;
            return this;
        }

        public String getCountry() {
            return country;
        }

        public Builder withCountry(String country) {
            this.country = country;
            return this;
        }

        public String getAdministrativeArea() {
            return administrativeArea;
        }

        public Builder withAdministrativeArea(String administrativeArea) {
            this.administrativeArea = administrativeArea;
            return this;
        }

        public String getCity() {
            return city;
        }

        public Builder withCity(String city) {
            this.city = city;
            return this;
        }

        public String getStreet() {
            return street;
        }

        public Builder withStreet(String street) {
            this.street = street;
            return this;
        }

        public String getHouse() {
            return house;
        }

        public Builder withHouse(String house) {
            this.house = house;
            return this;
        }

        public String getText() {
            return text;
        }

        public Builder withText(String text) {
            this.text = text;
            return this;
        }

        public String getName() {
            return name;
        }

        public Builder withName(String name) {
            this.name = name;
            return this;
        }

        public BigDecimal getX() {
            return x;
        }

        public Builder withX(BigDecimal x) {
            this.x = x;
            return this;
        }

        public Builder withX(double x) {
            this.x = BigDecimal.valueOf(x);
            return this;
        }

        public BigDecimal getY() {
            return y;
        }

        public Builder withY(BigDecimal y) {
            this.y = y;
            return this;
        }

        public Builder withY(double y) {
            this.y = BigDecimal.valueOf(y);
            return this;
        }

        public BigDecimal getX1() {
            return x1;
        }

        public Builder withX1(BigDecimal x1) {
            this.x1 = x1;
            return this;
        }

        public Builder withX1(double x1) {
            this.x1 = BigDecimal.valueOf(x1);
            return this;
        }

        public BigDecimal getY1() {
            return y1;
        }

        public Builder withY1(BigDecimal y1) {
            this.y1 = y1;
            return this;
        }

        public Builder withY1(double y1) {
            this.y1 = BigDecimal.valueOf(y1);
            return this;
        }

        public BigDecimal getX2() {
            return x2;
        }

        public Builder withX2(BigDecimal x2) {
            this.x2 = x2;
            return this;
        }

        public Builder withX2(double x2) {
            this.x2 = BigDecimal.valueOf(x2);
            return this;
        }

        public BigDecimal getY2() {
            return y2;
        }

        public Builder withY2(BigDecimal y2) {
            this.y2 = y2;
            return this;
        }

        public Builder withY2(double y2) {
            this.y2 = BigDecimal.valueOf(y2);
            return this;
        }

        public Precision getPrecision() {
            return precision;
        }

        public Builder withPrecision(Precision precision) {
            this.precision = precision;
            return this;
        }

        public Kind getKind() {
            return kind;
        }

        public Builder withKind(Kind kind) {
            this.kind = kind;
            return this;
        }

        public List<AddressComponent> getComponents() {
            return components;
        }

        public Builder withComponents(List<AddressComponent> components) {
            this.components = components;
            return this;
        }

        public GeoObject build() {
            return new GeoObject(
                    geoId,
                    houses,
                    country,
                    administrativeArea,
                    city,
                    street,
                    house,
                    text,
                    name,

                    x, y,
                    x1, y1,
                    x2, y2,

                    precision,
                    kind,

                    components);
        }
    }
}
