package ru.yandex.direct.geosuggest.client

import com.fasterxml.jackson.module.kotlin.convertValue
import com.fasterxml.jackson.module.kotlin.jacksonObjectMapper
import ru.yandex.direct.asynchttp.ParallelFetcherFactory
import ru.yandex.direct.geosuggest.client.model.GeoSuggestResponse
import ru.yandex.direct.http.smart.core.Smart
import ru.yandex.direct.http.smart.error.ErrorUtils

open class GeoSuggestClient(
    parallelFetcherFactory: ParallelFetcherFactory,
    geoSuggestApiUrl: String,
) {

    private val geoSuggestApi: GeoSuggestApi

    init {
        geoSuggestApi = Smart.builder()
            .withParallelFetcherFactory(parallelFetcherFactory)
            .withProfileName("geosuggest_api")
            .withBaseUrl(geoSuggestApiUrl)
            .build()
            .create(GeoSuggestApi::class.java)
    }

    // Returns list of regions used in Direct with names starting from text https://wiki.yandex-team.ru/suggest/geo/
    fun getSuggestedRegions(
        limit: Int,
        clientId: String,
        basesTypes: String,
        highlightMatching: Boolean,
        responseVersion: Int,
        latLong: String,
        lang: String,
        text: String
    ): GeoSuggestResponse {
        val call = geoSuggestApi.getSuggestedRegions(
            limit,
            clientId,
            basesTypes,
            "", // disable jsonp callback, receive json
            highlightMatching.compareTo(false),
            responseVersion,
            latLong,
            lang,
            text
        )
        val res = call.execute()

        ErrorUtils.checkResultForErrors(res, ::GeoSuggestClientException)
        if (res.success.isNullOrEmpty()) {
            throw GeoSuggestClientException("Handle returned empty response")
        }

        return jacksonObjectMapper()
            .convertValue(res.success[1]) // first element of the response array is query text, skip it
    }
}
