package ru.yandex.direct.graphite;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Асинхронная отправка метрик в Graphite с буфферизацией
 * При перелолнении буфера пишем в лог сообщение и теряем метрику
 */
@ParametersAreNonnullByDefault
public class GraphiteAsyncClient implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(GraphiteAsyncClient.class);
    private static final Duration ITERATION_DURATION = Duration.ofMillis(100);
    private static final int SEND_CHUNK_SIZE = 1_000;

    private final ArrayBlockingQueue<GraphiteMetric> buffer;
    private final GraphiteClient graphiteClient;
    private final boolean enabled;
    // флаг, по которому остановится фоновый поток отправки данных
    private volatile boolean bgStop = false;

    /**
     * @param capacity       - объём буффера
     * @param graphiteClient - синхронный клиент Graphite
     */
    public GraphiteAsyncClient(int capacity, GraphiteClient graphiteClient) {
        this(capacity, graphiteClient, true);
    }

    public GraphiteAsyncClient(int capacity, GraphiteClient graphiteClient, boolean enabled) {
        this.buffer = new ArrayBlockingQueue<>(capacity);
        this.graphiteClient = checkNotNull(graphiteClient);
        this.enabled = enabled;
    }

    @PostConstruct
    public void init() {
        if (!enabled) {
            logger.info("GraphiteAsyncClient disabled.");
            return;
        }

        Thread thread = new Thread(() -> {
            while (true) {
                if (Thread.currentThread().isInterrupted() || (bgStop && buffer.isEmpty())) {
                    break;
                }
                try {
                    GraphiteMetric first = buffer.poll(ITERATION_DURATION.toNanos(), TimeUnit.NANOSECONDS);
                    if (first != null) {
                        List<GraphiteMetric> data = new ArrayList<>();
                        data.add(first);
                        buffer.drainTo(data, SEND_CHUNK_SIZE - 1);
                        graphiteClient.send(data);
                    }
                } catch (InterruptedException e) {
                    logger.error("caught exception, break", e);
                    Thread.currentThread().interrupt();
                } catch (Exception e) {
                    logger.warn("caught exception, continue", e);
                }
            }
            logger.info("finish thread");
        });
        thread.setDaemon(true);
        thread.setName("graphite-async-1");
        thread.start();
    }

    /**
     * Добавить метрику в буфер
     */
    public boolean add(GraphiteMetric metric) {
        if (!enabled) {
            return true;
        }
        if (!buffer.offer(metric)) {
            logger.error("Can't add metric to graphite batcher - queue is full: {}", metric);
            return false;
        } else {
            return true;
        }
    }

    /**
     * Добавить метрики в буфер, если места нет - ждать не более timeout
     */
    public boolean add(List<GraphiteMetric> data, Duration timeout) {
        if (!enabled) {
            return true;
        }
        long startTime = System.nanoTime();
        for (int i = 0; i < data.size(); i++) {
            GraphiteMetric metric = data.get(i);
            // fast-path - скорее всего в буфере есть место
            if (!buffer.offer(metric)) {
                long elementTimeout = timeout.toNanos() - (System.nanoTime() - startTime);
                boolean res = false;
                if (elementTimeout > 0) {
                    try {
                        res = buffer.offer(metric, elementTimeout, TimeUnit.NANOSECONDS);
                    } catch (InterruptedException e) {
                        logger.error("Can't add metrics to queue, interrupted", e);
                        Thread.currentThread().interrupt();
                    }
                }
                if (!res) {
                    logger.error("Can't add metrics to queue({} metrics left), timeout", data.size() - i);
                    return false;
                }
            }
        }
        return true;
    }

    @Override
    @PreDestroy
    public void close() {
        bgStop = true;
    }
}
