package ru.yandex.direct.graphite;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.SocketChannel;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.List;
import java.util.concurrent.TimeoutException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Синхронный клиент для отпрвки данных в Graphite
 */
public class GraphiteClient {
    private static final Logger logger = LoggerFactory.getLogger(GraphiteClient.class);

    private final String host;
    private final int port;
    private final Duration timeout;

    /**
     * @param host    - сервер (обычно localhost для graphite-sender-а)
     * @param port    - порт (обычно 42000)
     * @param timeout - таймаут на все операции вместе - коннект + отправка данных
     */
    @SuppressWarnings("CheckReturnValue")
    public GraphiteClient(String host, int port, Duration timeout) {
        checkNotNull(host);
        checkArgument(port > 0 && port < 65536, "Incorrect port %s", port);
        checkArgument(timeout.toMillis() > 0, "Timeout must be greater than 0: %s", timeout);

        this.host = host;
        this.port = port;
        this.timeout = timeout;
    }

    /**
     * Синхронная отправка списка метрик
     *
     * @param metrics - список метрик
     * @throws GraphiteException - если случился таймаут или сетевая ошибка
     */
    public void send(List<GraphiteMetric> metrics) {
        logger.trace("send {} metrics", metrics.size());
        if (metrics.isEmpty()) {
            return;
        }

        StringBuilder stringBuilder = new StringBuilder();
        metrics.forEach(m -> stringBuilder.append(m).append("\n"));
        String request = stringBuilder.toString();
        logger.trace("data:\n{}", request);

        try (TraceProfile profile = Trace.current().profile("graphite:send")) {
            send(request.getBytes(StandardCharsets.UTF_8));
        } catch (IOException | TimeoutException e) {
            throw new GraphiteException(e.getMessage(), e);
        }
    }

    /**
     * Синхронная отправка метрик из буффера
     *
     * @param metricsBuffer - буффер с метриками
     * @throws GraphiteException - если случился таймаут или сетевая ошибка
     */
    public void send(GraphiteMetricsBuffer metricsBuffer) {
        send(metricsBuffer.drain());
    }

    private void send(byte[] data) throws IOException, TimeoutException {
        long border = System.nanoTime() + timeout.toNanos();
        InetSocketAddress addr = new InetSocketAddress(host, port);

        ByteBuffer bb = ByteBuffer.wrap(data);

        try (SocketChannel ch = SocketChannel.open();
             Selector sel = Selector.open();
        ) {
            ch.configureBlocking(false);
            ch.register(sel, SelectionKey.OP_CONNECT | SelectionKey.OP_WRITE);
            ch.connect(addr);

            boolean connected = false;
            while (true) {
                long remainingTime = border - System.nanoTime();
                if (remainingTime <= 0 || bb.remaining() <= 0) {
                    break;
                }
                sel.select(remainingTime);

                for (SelectionKey selectionKey : sel.selectedKeys()) {
                    if (!connected && selectionKey.isConnectable()) {
                        ch.finishConnect();
                        selectionKey.interestOps(SelectionKey.OP_WRITE);
                        connected = true;
                    } else if (selectionKey.isWritable()) {
                        ch.write(bb);
                    }
                }
            }

            if (bb.remaining() > 0) {
                throw new TimeoutException("Can't write to graphite with timeout " + timeout);
            }
        }
    }
}
