package ru.yandex.direct.graphite;

import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Одна точка метрики, отправляемой в Graphite
 */
@ParametersAreNonnullByDefault
public class GraphiteMetric {

    private final String metric;
    private final double value;
    private final long timestamp;

    public GraphiteMetric(String metric, double value) {
        this(metric, value, System.currentTimeMillis() / 1_000);
    }

    /**
     * @param metric    - название метрики, например one_min.xxx.yyy
     * @param value     - значение метрики
     * @param timestamp - на какое время записать значение
     */
    @SuppressWarnings("CheckReturnValue")
    public GraphiteMetric(String metric, double value, long timestamp) {
        checkNotNull(metric);
        checkArgument(GraphiteUtils.METRIC_PATTERN.matcher(metric).matches(), "Incorrect metric name '%s'", metric);
        this.metric = metric;
        this.value = value;
        this.timestamp = timestamp;
    }

    public String getMetric() {
        return metric;
    }

    public double getValue() {
        return value;
    }

    public long getTimestamp() {
        return timestamp;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        GraphiteMetric that = (GraphiteMetric) o;
        return Double.compare(that.value, value) == 0 &&
                timestamp == that.timestamp &&
                Objects.equals(metric, that.metric);
    }

    @Override
    public int hashCode() {
        return Objects.hash(metric, value, timestamp);
    }

    /**
     * @return представление метрики в формате, пригодном для отправки в Graphite:
     * "metric value timestamp"
     */
    @Override
    public String toString() {
        return metric + " " + value + " " + timestamp;
    }
}
