package ru.yandex.direct.graphite;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import static java.util.Collections.emptyList;

/**
 * Помощник для формирования списка метрик с общим префиксом.
 * Имена метрик формируются из частей, объединенных точками.
 * <p>
 * Например, список из следующих метрик:
 * <ul>
 * <li>new GraphiteMetric("junk.my_cool_stat.key1.stat1", 1);</li>
 * <li>new GraphiteMetric("junk.my_cool_stat.key1.stat2", 2);</li>
 * <li>new GraphiteMetric("junk.my_cool_stat.total", 100);</li>
 * </ul>
 * можно сформировать так:
 * <pre> {@code
 * GraphiteMetricsBuffer metrics = new GraphiteMetricsBuffer("junk.my_cool_stat");
 * metrics.with("key1").add("stat1", 1).add("stat2", 2);
 * metrics.add("total", 100);
 * List<GraphiteMetric> list = metrics.drain();
 * }</pre>
 */
@ParametersAreNonnullByDefault
public class GraphiteMetricsBuffer {
    private final String prefix;
    private final List<GraphiteMetric> buffer;

    /**
     * Создать пустой буффер со строковым префиксом
     *
     * @param prefix - префикс-строка к именам всех метрик
     */
    public GraphiteMetricsBuffer(String prefix) {
        this.buffer = new ArrayList<>();
        this.prefix = prefix;
    }

    /**
     * Создать пустой буффер с префиксом, составленным из частей
     *
     * @param prefixParts - строки, из которых будет сформирован префикс
     */
    public GraphiteMetricsBuffer(String... prefixParts) {
        this(joinParts(prefixParts));
    }

    private GraphiteMetricsBuffer(String prefix, List<GraphiteMetric> buffer) {
        this.buffer = buffer;
        this.prefix = prefix;
    }

    /**
     * Получить новый буффер с накопленными метриками и дополненным префиксом
     *
     * @param parts - строки, которые будут добавлены к текущему префиксу
     * @return новый буфер
     */
    public GraphiteMetricsBuffer with(String... parts) {
        return new GraphiteMetricsBuffer(path(parts), buffer);
    }

    /**
     * Получить новый буффер с накопленными метриками и дополненным префиксом.
     * Все части пути нормализуются.
     *
     * @param parts - строки, которые будут добавлены к текущему префиксу
     * @return новый буфер
     */
    public GraphiteMetricsBuffer withSafely(String... parts) {
        return new GraphiteMetricsBuffer(pathSafely(parts), buffer);
    }

    /**
     * Добавляет в буффер метрику с именем - текущим префиксом
     *
     * @param value     - значение метрики
     * @param timestamp - на какое время записать значение
     * @return текущий буффер
     */
    public GraphiteMetricsBuffer add(double value, long timestamp) {
        buffer.add(new GraphiteMetric(path(), value, timestamp));
        return this;
    }

    /**
     * Добавляет в буффер метрику с именем - текущим префиксом
     *
     * @param value - значение метрики
     * @return текущий буффер
     */
    public GraphiteMetricsBuffer add(double value) {
        buffer.add(new GraphiteMetric(path(), value));
        return this;
    }

    /**
     * Добавляет в буффер метрику с именем, составленным из текущего префикса и переданной части
     *
     * @param part  - часть названия метрики
     * @param value - значение метрики
     * @return текущий буффер
     */
    public GraphiteMetricsBuffer add(String part, double value) {
        buffer.add(new GraphiteMetric(path(part), value));
        return this;
    }

    /**
     * Добавляет в буффер метрику с именем, составленным из текущего префикса и переданной части
     *
     * @param part      - часть названия метрики
     * @param value     - значение метрики
     * @param timestamp - на какое время записать значение
     * @return текущий буффер
     */
    public GraphiteMetricsBuffer add(String part, double value, long timestamp) {
        buffer.add(new GraphiteMetric(path(part), value, timestamp));
        return this;
    }

    /**
     * Получить список метрик и очистить буфер
     *
     * @return {@link List<GraphiteMetric>} накопленные метрики
     */
    public List<GraphiteMetric> drain() {
        if (buffer.isEmpty()) {
            return emptyList();
        }
        List<GraphiteMetric> ret = new ArrayList<>(buffer);
        buffer.clear();
        return ret;
    }

    /**
     * @return - количество сетрик в буфере
     */
    public int size() {
        return buffer.size();
    }

    /**
     * Стрингифицированный путь с учётом префикса
     */
    public String path(String... parts) {
        if (prefix.isEmpty()) {
            return joinParts(parts);
        } else if (parts.length == 0) {
            return prefix;
        } else {
            return joinParts(prefix, joinParts(parts));
        }
    }

    /**
     * Стрингифицированный путь с учётом префикса.
     * Все части пути нормализуются.
     */
    public String pathSafely(String... parts) {
        return path(Arrays.stream(parts).map(GraphiteUtils::normalizeMetricPart).toArray(String[]::new));
    }

    private static String joinParts(String... parts) {
        return Stream.of(parts).collect(Collectors.joining("."));
    }

    @Override
    public String toString() {
        final StringBuilder sb = new StringBuilder("GraphiteMetricsBuffer{");
        sb.append("prefix='").append(prefix).append('\'');
        sb.append(", buffer=").append(buffer);
        sb.append('}');
        return sb.toString();
    }
}
