package ru.yandex.direct.grid.schemagen;

import java.io.FileWriter;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.sql.SQLException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.springframework.context.annotation.AnnotationConfigApplicationContext;

import ru.yandex.direct.dbschemagen.DbSchemaGen;
import ru.yandex.direct.dbschemagen.DbSchemaGenParams;
import ru.yandex.direct.grid.schemagen.configuration.GridSchemaGenConfiguration;
import ru.yandex.direct.grid.schemagen.sqlgen.TableSqlGenerator;
import ru.yandex.direct.grid.schemagen.sqlgen.YtSqlGenerator;
import ru.yandex.direct.jcommander.ParserWithHelp;
import ru.yandex.direct.mysql.ytsync.common.components.SyncStatesTable;
import ru.yandex.direct.mysql.ytsync.task.config.DirectYtSyncConfig;

/**
 * Инструмент для сбора jooq-схем из двух мест - описаний таблиц в mysql-direct-yt-sync и YT-схем таблиц со
 * статистикой БК
 */
public class GridSchemaGenTool {
    private static final String SCHEMA_PACKAGE = "ru.yandex.direct.grid.schema";

    private static final String GRID_SYNC_STATES_TABLE_NAME = "Direct_Grid_Mysql_Sync_States";

    public static void main(String[] args) throws Exception {
        GridSchemaGenToolParams params = new GridSchemaGenToolParams();
        ParserWithHelp.parse(GridSchemaGenTool.class.getCanonicalName(), args, params);
        runInScope(params.generatedCodePath);
    }

    private static void runInScope(String outputPath) throws InterruptedException, SQLException, IOException {
        try (AnnotationConfigApplicationContext ctx = new AnnotationConfigApplicationContext(
                GridSchemaGenConfiguration.class)) {
            TableSqlGenerator tableSqlGenerator = ctx.getBean(TableSqlGenerator.class);
            DirectYtSyncConfig ytSyncConfig = ctx.getBean(DirectYtSyncConfig.class);
            Map<String, String> schemas = new HashMap<>(tableSqlGenerator.getSchemas("direct", ytSyncConfig));

            addCustomTables(schemas);

            YtSqlGenerator ytSqlGenerator = ctx.getBean(YtSqlGenerator.class);
            schemas.putAll(ytSqlGenerator.getYabsSchemas("bs"));
            schemas.putAll(ytSqlGenerator.getRecommendationsSchemas("recommendations"));
            schemas.putAll(ytSqlGenerator.getMobileSchema("direct"));
            schemas.putAll(ytSqlGenerator.getLogsSchema("test_logs"));
            schemas.putAll(ytSqlGenerator.getCounterByDomainSchema());
            schemas.putAll(ytSqlGenerator.getAllCountersByDomainSchema());
            schemas.putAll(ytSqlGenerator.getCountersByDomainHitLogSchema());
            schemas.putAll(ytSqlGenerator.getCalltrackingNumberClicksSchema());
            schemas.putAll(ytSqlGenerator.getCalltrackingNumbersWithoutChangesSchema());
            schemas.putAll(ytSqlGenerator.getCalltrackingExternalByDomainSchema());
            schemas.putAll(ytSqlGenerator.getBrandSurveysSchemas());
            schemas.putAll(ytSqlGenerator.getCategoriesWhitelistSchemas());
            schemas.putAll(ytSqlGenerator.getCategoriesBlockedDomainsSchemas());
            schemas.putAll(ytSqlGenerator.getSitelinksSchemas("import_from_bno_sitelinks"));
            schemas.putAll(ytSqlGenerator.getCommunicationPlatformMessagesSchemas("CommunicationsChannel"));
            schemas.putAll(ytSqlGenerator.getCommunicationEventConfigSchemas("EventConfiguration"));
            schemas.putAll(ytSqlGenerator.getCommunicationEventVersionsConfigSchemas("EventVersionConfiguration"));
            schemas.putAll(ytSqlGenerator.getCommunicationSlotsConfigSchemas("CommunicationSlotConfiguration"));
            schemas.putAll(ytSqlGenerator.getSuggestConversionPriceSchema());
            schemas.putAll(ytSqlGenerator.getMetrikaConvAdGoalsSchema());
            schemas.putAll(ytSqlGenerator.getBlrtSchemas());
            schemas.putAll(ytSqlGenerator.getConvMainCountersSchema());

            generate(outputPath, schemas);
        }
    }

    private static void addCustomTables(Map<String, String> schemas) {
        String tableSchema = TableSqlGenerator.getTableSchema(SyncStatesTable.SCHEMA, GRID_SYNC_STATES_TABLE_NAME);
        schemas.put(GRID_SYNC_STATES_TABLE_NAME, tableSchema);
    }

    private static Path createSchemaDir(Map<String, String> schemas) throws IOException {
        Path tempDir = Files.createTempDirectory("ytschema");
        tempDir = Files.createDirectory(tempDir.resolve("yt"));
        for (Map.Entry<String, String> entry : schemas.entrySet()) {
            Path path = Files.createFile(tempDir.resolve(entry.getKey() + ".schema.sql"));
            try (FileWriter writer = new FileWriter(path.toFile())) {
                writer.write(entry.getValue());
            }
        }
        return tempDir;
    }

    private static void generate(String outputPath, Map<String, String> schemas)
            throws InterruptedException, SQLException, IOException {
        Path schemaSqlDir = createSchemaDir(schemas);
        DbSchemaGenParams params = new DbSchemaGenParams();
        params.generatedCodePath = outputPath;
        params.guessArtificialBigintColumns = false;
        params.packageNamespace = SCHEMA_PACKAGE;

        DbSchemaGen.runWithDocker(params, Collections.singletonList(schemaSqlDir), true);
    }

}
