package ru.yandex.direct.grid.schemagen.sqlgen;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.ws.rs.NotFoundException;

import com.google.common.base.Strings;
import com.google.common.collect.Iterables;
import one.util.streamex.EntryStream;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.mobilecontent.MobileContentYtTable;
import ru.yandex.direct.common.mobilecontent.MobileContentYtTablesConfig;
import ru.yandex.direct.ytcomponents.config.DirectYtDynamicConfig;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.direct.ytwrapper.model.YtTable;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;

import static java.util.Collections.singletonList;
import static ru.yandex.direct.grid.schemagen.util.YtUtil.mysqlTypeOf;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class YtSqlGenerator {
    private static final String MOBILE_STORE = "gplay";
    private static final String MOBILE_TABLE_NAME = "ExtdataMobile_direct";

    private final YtProvider ytProvider;
    private final DirectYtDynamicConfig dynamicConfig;
    private final MobileContentYtTablesConfig mobileConfig;

    @Autowired
    public YtSqlGenerator(YtProvider ytProvider, DirectYtDynamicConfig dynamicConfig,
                          MobileContentYtTablesConfig mobileConfig) {
        this.ytProvider = ytProvider;
        this.dynamicConfig = dynamicConfig;
        this.mobileConfig = mobileConfig;
    }

    public Map<String, String> getYabsSchemas(String postfix) {
        var dynamicTables = dynamicConfig.tables();
        var yabsStatTables = dynamicTables.yabsStat();
        var yabsDictTables = dynamicTables.yabsDict();
        var tables = Arrays.asList(
                yabsStatTables.taxHistoryTablePath(),
                yabsStatTables.phrasesTablePath(),
                yabsStatTables.ordersTablePath(),
                yabsStatTables.conversionsTablePath(),
                yabsStatTables.orderStatDayTablePath(),
                yabsStatTables.orderStatFraudTablePath(),
                yabsStatTables.phraseGoalsTablePath(),
                yabsStatTables.orderGoalsTablePath(),
                yabsStatTables.orderStatTablePath(),
                yabsStatTables.mobileAppStatTablePath(),
                yabsStatTables.mobileEventsStatTablePath(),
                yabsDictTables.orderInfoDictTablePath(),
                yabsDictTables.currencyDictTablePath(),
                yabsDictTables.oldPhraseIdsTablePath(),
                yabsStatTables.assetsStatTablePath(),
                yabsStatTables.offerStatTablePath(),
                yabsStatTables.offerAttributesTablePath()
        );

        var schemas = getSchemas(postfix, tables);
        schemas.putAll(getBrandSafetyFiltrationsSchemas(postfix));
        schemas.putAll(getBrandSafetyStatSchemas(postfix));
        schemas.putAll(getBrandSafetyCategoryStatSchemas(postfix));
        return schemas;
    }

    public Map<String, String> getBrandSurveysSchemas() {
        var tables = singletonList(dynamicConfig.tables().direct().brandSurveysTablePath());
        return getSchemas("", tables, dynamicConfig.getBrandSurveysClusters());
    }

    public Map<String, String> getCategoriesWhitelistSchemas() {
        var tables = singletonList(dynamicConfig.tables().direct().categoriesWhitelistTablePath());
        return getSchemas("", tables, dynamicConfig.getCategoriesClusters());
    }

    public Map<String, String> getCategoriesBlockedDomainsSchemas() {
        var tables = singletonList(dynamicConfig.tables().direct().categoriesBlockedDomainsTablePath());
        return getSchemas("", tables, dynamicConfig.getCategoriesClusters());
    }

    public Map<String, String> getBrandSafetyFiltrationsSchemas(String postfix) {
        var tables = singletonList(dynamicConfig.tables().yabsStat().brandSafetyFiltrationsTablePath());
        return getSchemas(postfix, tables, dynamicConfig.getStatClusters());
    }

    public Map<String, String> getBrandSafetyStatSchemas(String postfix) {
        var tables = singletonList(dynamicConfig.tables().yabsStat().brandSafetyStatTablePath());
        return getSchemas(postfix, tables, dynamicConfig.getStatClusters());
    }

    public Map<String, String> getBrandSafetyCategoryStatSchemas(String postfix) {
        var tables = singletonList(dynamicConfig.tables().yabsStat().brandSafetyCategoryStatTablePath());
        return getSchemas(postfix, tables, dynamicConfig.getStatClusters());
    }

    public Map<String, String> getRecommendationsSchemas(String postfix) {
        var tables = singletonList(dynamicConfig.tables().recommendations().recommendationsTablePath());
        return getSchemas(postfix, tables);
    }

    public Map<String, String> getSitelinksSchemas(String name) {
        var table = dynamicConfig.tables().sitelinks().sitelinksImportTablePath();
        return getSchema(name, table, dynamicConfig.getSitelinksImportClusters());
    }

    public Map<String, String> getMobileSchema(String postfix) {
        // Схема для генерации таблицы EXTDATAMOBILEDIRECT берется с учетом ее идентичности для всех сторов
        MobileContentYtTable table = mobileConfig.getShopInfo(MOBILE_STORE).orElseThrow(NotFoundException::new);
        var tablePath = table.getTable();
        return getSchema(MOBILE_TABLE_NAME, tablePath, mapList(table.getClusters(), YtCluster::parse));
    }

    public Map<String, String> getLogsSchema(String postfix) {
        var tables = singletonList(dynamicConfig.tables().direct().bsExportlogsTablePath());
        return getSchemas(postfix, tables, dynamicConfig.getTestLogsClusters());
    }

    public Map<String, String> getCounterByDomainSchema() {
        var tables = singletonList(dynamicConfig.tables().direct().counterByDomainTablePath());
        return getSchemas("", tables, dynamicConfig.getCounterByDomainClusters());
    }

    public Map<String, String> getAllCountersByDomainSchema() {
        var tables = singletonList(dynamicConfig.tables().direct().allCountersByDomainTablePath());
        return getSchemas("", tables, dynamicConfig.getAllCountersByDomainClusters());
    }

    public Map<String, String> getCountersByDomainHitLogSchema() {
        var tables = singletonList(dynamicConfig.tables().direct().countersByDomainHitLogTablePath());
        return getSchemas("", tables, dynamicConfig.getCountersByDomainHitLogClusters());
    }

    public Map<String, String> getCalltrackingNumberClicksSchema() {
        var tables = singletonList(dynamicConfig.tables().direct().calltrackingNumberClicksTablePath());
        return getSchemas("", tables, dynamicConfig.getCalltrackingNumberClicksClusters());
    }

    public Map<String, String> getCalltrackingNumbersWithoutChangesSchema() {
        var tables =
                singletonList(dynamicConfig.tables().direct().calltrackingPhonesWithoutReplacementsTablePath());
        return getSchemas("", tables, dynamicConfig.getCalltrackingPhonsWithoutReplacementsClusters());
    }

    public Map<String, String> getCalltrackingExternalByDomainSchema() {
        var tables = singletonList(dynamicConfig.tables().direct().calltrackingExternalByDomainTablePath());
        return getSchemas("", tables, dynamicConfig.getCalltrackingExternalByDomainsClusters());
    }

    public Map<String, String> getCommunicationPlatformMessagesSchemas(String name) {
        var table = dynamicConfig.tables().communicationPlatform().messagesTablePath();
        return getSchema(name, table, dynamicConfig.getCommunicationPlatformClusters());
    }

    public Map<String, String> getCommunicationEventConfigSchemas(String name) {
        var tables = dynamicConfig.tables().communicationPlatform();
        return getSchema(name, tables.eventConfigTablePath(), tables.readConfigClusters());
    }

    public Map<String, String> getCommunicationEventVersionsConfigSchemas(String name) {
        var tables = dynamicConfig.tables().communicationPlatform();
        return getSchema(name, tables.eventVersionConfigTablePath(), tables.readConfigClusters());
    }

    public Map<String, String> getCommunicationSlotsConfigSchemas(String name) {
        var tables = dynamicConfig.tables().communicationPlatform();
        return getSchema(name, tables.slotConfigTablePath(), tables.readConfigClusters());
    }

    public Map<String, String> getSuggestConversionPriceSchema() {
        var table = singletonList(dynamicConfig.tables().direct().suggestConversionPriceTablePath());
        return getSchemas("", table, dynamicConfig.getSuggestConversionPriceClusters());
    }

    public Map<String, String> getMetrikaConvAdGoalsSchema() {
        var table = singletonList(dynamicConfig.tables().metrikaConvAdGoals().metrikaConvAdGoalsTablePath());
        return getSchemas("", table, dynamicConfig.getMetrikaConvAdGoalsClusters());
    }

    public Map<String, String> getBlrtSchemas() {
        var blrtTables = dynamicConfig.tables().blrt();
        var blrtClusters = dynamicConfig.getBlrtClusters();
        return EntryStream.<String, String>empty()
                .append(getSchema("feed_to_tasks_dyn", blrtTables.dynFeedToTasksTablePath(), blrtClusters))
                .append(getSchema("feed_to_tasks_perf", blrtTables.perfFeedToTasksTablePath(), blrtClusters))
                .toMap();
    }

    public Map<String, String> getConvMainCountersSchema() {
        var table = singletonList(dynamicConfig.tables().convMainCountersTables().convMainCountersTablePath());
        return getSchemas("", table, dynamicConfig.getConvMainCountersClusters());
    }

    Map<String, String> getSchemas(String postfix, List<String> tables) {
        return getSchemas(postfix, tables, dynamicConfig.getClusters());
    }

    private Map<String, String> getSchema(String name, String table, Collection<YtCluster> clusters) {
        var ytOperator = ytProvider.getOperator(
                Objects.requireNonNull(Iterables.getFirst(clusters, null)));
        return Map.of(name, getTableSchema(table, name, ytOperator).trim());
    }

    private Map<String, String> getSchemas(String postfix, List<String> tables, Collection<YtCluster> clusters) {
        var ytOperator = ytProvider.getOperator(
                Objects.requireNonNull(Iterables.getFirst(clusters, null)));
        var schemas = new HashMap<String, String>();
        for (var table : tables) {
            var name = new YtTable(table).ypath().name();
            var tableName = Strings.isNullOrEmpty(postfix) ? name : String.format("%s_%s", name, postfix).replace('-', '_');
            schemas.put(tableName, getTableSchema(table, tableName, ytOperator).trim());
        }
        return schemas;
    }

    private String getTableSchema(String path, String tableName, YtOperator ytOperator) {
        YtTable ytTable = new YtTable(path);
        List<YTreeMapNode> schema = ytOperator.getSchema(ytTable);

        StringBuilder sb = new StringBuilder();
        sb.append("CREATE TABLE ").append(tableName).append(" (\n");

        List<String> columns = new ArrayList<>();
        for (YTreeMapNode node : schema) {
            columns.add("  " + node.getString("name") + " " + mysqlTypeOf(
                    node.getString("type")));
        }
        sb.append(String.join(",\n", columns));
        sb.append("\n);\n");

        return sb.toString();
    }
}
