package ru.yandex.direct.hourglass.ydb.monitoring;

import java.util.concurrent.TimeUnit;

import com.yandex.ydb.table.TableClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.hourglass.MonitoringWriter;
import ru.yandex.direct.hourglass.implementations.ThreadsHierarchy;
import ru.yandex.direct.hourglass.implementations.internal.SystemThread;
import ru.yandex.monlib.metrics.primitives.GaugeInt64;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

public class YdbSessionMonitoring implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(YdbSessionMonitoring.class);
    private final TableClient tableClient;
    private final GaugeInt64 acquiredCount;
    private final GaugeInt64 pendingAcquireCount;
    private final GaugeInt64 idleCount;
    private final GaugeInt64 disconnectedCount;
    private final SystemThread logStatsThread;

    public YdbSessionMonitoring(MetricRegistry metricRegistry, MonitoringWriter monitoringWriter,
                                ThreadsHierarchy threadsHierarchy,
                                TableClient tableClient) {
        this.tableClient = tableClient;
        acquiredCount = metricRegistry.gaugeInt64("ydb_sessions_acquired_count");
        pendingAcquireCount = metricRegistry.gaugeInt64("ydb_sessions_pending_acquired_count");
        idleCount = metricRegistry.gaugeInt64("ydb_sessions_idle_count");
        disconnectedCount = metricRegistry.gaugeInt64("ydb_sessions_disconnected_count");
        this.logStatsThread = new SystemThread(this::monitor, threadsHierarchy.getSystemThreadFactory(), "session" +
                "-pool-monitoring", 1, TimeUnit.SECONDS, monitoringWriter);
        logStatsThread.start();
    }

    public void monitor() {
        var sessionPoolStats = tableClient.getSessionPoolStats();
        acquiredCount.set(sessionPoolStats.getAcquiredCount());
        pendingAcquireCount.set(sessionPoolStats.getPendingAcquireCount());
        idleCount.set(sessionPoolStats.getIdleCount());
        disconnectedCount.set(sessionPoolStats.getDisconnectedCount());

        logger.info("Acquired count {}, pending acquire count {}, idle count {}, disconnected count {}",
                sessionPoolStats.getAcquiredCount(),
                sessionPoolStats.getPendingAcquireCount(), sessionPoolStats.getIdleCount(),
                sessionPoolStats.getDisconnectedCount());
    }

    @Override
    public void close() {
        logStatsThread.stop();
    }
}
