package ru.yandex.direct.hourglass.ydb.schedulerinstances;

import java.time.Duration;
import java.time.Instant;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Supplier;

import javax.annotation.Nullable;

import com.yandex.ydb.table.SessionRetryContext;
import com.yandex.ydb.table.TableClient;

import ru.yandex.direct.hourglass.InstanceId;
import ru.yandex.direct.hourglass.InstanceMeta;
import ru.yandex.direct.hourglass.client.SchedulerInstance;
import ru.yandex.direct.hourglass.implementations.InstanceIdImpl;
import ru.yandex.direct.hourglass.implementations.updateschedule.SchedulerInstanceImpl;
import ru.yandex.direct.hourglass.updateschedule.SchedulerInstancesRepository;
import ru.yandex.direct.ydb.YdbPath;
import ru.yandex.direct.ydb.builder.predicate.Predicate;
import ru.yandex.direct.ydb.builder.querybuilder.QueryBuilder;
import ru.yandex.direct.ydb.client.YdbClient;

import static com.yandex.ydb.table.values.PrimitiveType.bool;
import static ru.yandex.direct.hourglass.ydb.storage.Tables.SCHEDULER_INSTANCES;
import static ru.yandex.direct.ydb.builder.expression.AggregateExpression.count;
import static ru.yandex.direct.ydb.builder.expression.CaseExpression.caseWhen;
import static ru.yandex.direct.ydb.builder.querybuilder.DeleteBuilder.deleteFrom;
import static ru.yandex.direct.ydb.builder.querybuilder.InsertBuilder.insertInto;
import static ru.yandex.direct.ydb.builder.querybuilder.OrderByBuilder.OrderType.DESC;
import static ru.yandex.direct.ydb.builder.querybuilder.SelectBuilder.select;
import static ru.yandex.direct.ydb.builder.querybuilder.UpdateBuilder.set;
import static ru.yandex.direct.ydb.builder.querybuilder.UpdateBuilder.update;
import static ru.yandex.direct.ydb.table.temptable.TempTable.tempTable;

public class YdbScheduleInstancesRepository implements SchedulerInstancesRepository {

    private static final Duration DEFAULT_INSTANCES_EXPIRATION = Duration.ofDays(1);
    private static final Duration QUERY_TIMEOUT = Duration.ofMinutes(1);
    private final Supplier<Predicate> isActivePredicate;
    private final Duration instancesExpiration;
    private final YdbPath path;
    private final YdbClient ydbClient;
    private final int leaderVotingLowerBound;

    public YdbScheduleInstancesRepository(TableClient tableClient, YdbPath db, Duration heartbeatExpiration,
                                          int leaderVotingLowerBound,
                                          Duration instancesExpiration) {

        this.isActivePredicate = () -> SCHEDULER_INSTANCES.HEARTBEAT_TIME.ge(Instant.now().minus(heartbeatExpiration));
        this.instancesExpiration = instancesExpiration;
        this.path = db;
        var sessionRetryContext = SessionRetryContext.create(tableClient).maxRetries(5).build();
        this.ydbClient = new YdbClient(sessionRetryContext, QUERY_TIMEOUT);
        this.leaderVotingLowerBound = leaderVotingLowerBound;
    }

    public YdbScheduleInstancesRepository(TableClient tableClient, YdbPath db, Duration heartbeatExpiration,
                                          int leaderVotingLowerBound) {
        this(tableClient, db, heartbeatExpiration, leaderVotingLowerBound, DEFAULT_INSTANCES_EXPIRATION);
    }

    @Override
    public void addInstance(InstanceId instanceId, String version, @Nullable InstanceMeta instanceMeta) {
        String meta = instanceMeta == null ? null : instanceMeta.getString();
        var insertValues = tempTable(SCHEDULER_INSTANCES.INSTANCE_ID,
                SCHEDULER_INSTANCES.HEARTBEAT_TIME,
                SCHEDULER_INSTANCES.IS_MAIN, SCHEDULER_INSTANCES.VERSION,
                SCHEDULER_INSTANCES.META)
                .createValues()
                .fill(instanceId.toString(), Instant.now(), true, version, meta);
        var queryAndParams = insertInto(SCHEDULER_INSTANCES)
                .selectAll()
                .from(insertValues)
                .queryAndParams(path);
        ydbClient.executeQuery(queryAndParams,
                "Failed to " + "insert new instance " + instanceId + " with version " + version);
    }

    @Override
    public void markInstanceAsNotMain(InstanceId instanceId) {
        setIsMain(instanceId, false);
    }

    @Override
    public void markInstanceAsMain(InstanceId instanceId) {
        setIsMain(instanceId, true);
    }

    private void setIsMain(InstanceId instanceId, boolean isMain) {
        QueryBuilder builder = update(SCHEDULER_INSTANCES, set(SCHEDULER_INSTANCES.IS_MAIN, isMain))
                .where(SCHEDULER_INSTANCES.INSTANCE_ID.eq(instanceId.toString()));
        var queryAndParams = builder.queryAndParams(path);
        ydbClient.executeQuery(queryAndParams, "Failed to mark instance as not main " + instanceId);
    }

    @Override
    public boolean isLeaderVersion(String currentVersion) {
        removeOldInstances();
        var countField = count().as("cnt");
        var queryBuilder = select(SCHEDULER_INSTANCES.VERSION, countField)
                .from(SCHEDULER_INSTANCES)
                .where(SCHEDULER_INSTANCES.IS_MAIN.eq(true).and(isActivePredicate.get()))
                .groupBy(SCHEDULER_INSTANCES.VERSION)
                .orderBy(DESC, countField);

        var queryAndParams = queryBuilder.queryAndParams(path);
        var dataQueryResultWrapper = ydbClient.executeQuery(queryAndParams, "Can't load instances for calculate " +
                "leader");
        var resultSetReader = dataQueryResultWrapper.getResultSet(0);
        resultSetReader.next();
        if (resultSetReader.isEmpty()) {
            return false;
        }
        var firstVersion = resultSetReader.getValueReader(SCHEDULER_INSTANCES.VERSION).getUtf8();
        long cnt1 = resultSetReader.getColumn("cnt").getUint64();
        long cnt2 = resultSetReader.next() ? resultSetReader.getColumn("cnt").getUint64() : 0;
        return cnt1 >= leaderVotingLowerBound && cnt1 != cnt2 && currentVersion.equals(firstVersion);

    }

    @Override
    public void pingInstance(InstanceId instanceId) {
        QueryBuilder builder = update(SCHEDULER_INSTANCES, set(SCHEDULER_INSTANCES.HEARTBEAT_TIME, Instant.now()))
                .where(SCHEDULER_INSTANCES.INSTANCE_ID.eq(instanceId.toString()));
        var queryAndParams = builder.queryAndParams(path);
        ydbClient.executeQuery(queryAndParams, "Failed to update instance heartbeat_time " + instanceId, true);
    }

    @Override
    public List<SchedulerInstance> getSchedulerInstancesInfo() {
        var isActiveField = caseWhen(isActivePredicate.get(), true, bool())
                .otherwise(false).as("isActive");
        var queryBuilder = select(SCHEDULER_INSTANCES.INSTANCE_ID,
                SCHEDULER_INSTANCES.VERSION,
                SCHEDULER_INSTANCES.IS_MAIN,
                SCHEDULER_INSTANCES.HEARTBEAT_TIME,
                SCHEDULER_INSTANCES.META,
                isActiveField)
                .from(SCHEDULER_INSTANCES);

        var queryAndParams = queryBuilder.queryAndParams(path);
        var dataQueryResultWrapper = ydbClient.executeQuery(queryAndParams, "Can't get scheduler instances info", true);

        var resultSetReader = dataQueryResultWrapper.getResultSet(0);
        List<SchedulerInstance> schedulerInstances = new ArrayList<>();
        while (resultSetReader.next()) {
            schedulerInstances.add(
                    new SchedulerInstanceImpl(
                            new InstanceIdImpl(resultSetReader.getValueReader(SCHEDULER_INSTANCES.INSTANCE_ID).getUtf8()),
                            resultSetReader.getValueReader(SCHEDULER_INSTANCES.VERSION).getUtf8(),
                            resultSetReader.getValueReader(SCHEDULER_INSTANCES.IS_MAIN).getBool(),
                            resultSetReader.getValueReader(SCHEDULER_INSTANCES.HEARTBEAT_TIME).getDatetime().toInstant(ZoneOffset.UTC),
                            resultSetReader.getValueReader(SCHEDULER_INSTANCES.META).getUtf8(),
                            resultSetReader.getValueReader(isActiveField).getBool()
                    )
            );
        }
        return schedulerInstances;
    }

    @Override
    public void removeInstance(InstanceId instanceId) {
        QueryBuilder builder = deleteFrom(SCHEDULER_INSTANCES)
                .where(SCHEDULER_INSTANCES.INSTANCE_ID.eq(instanceId.toString()));

        var queryAndParams = builder.queryAndParams(path);
        ydbClient.executeQuery(queryAndParams, "Failed to delete instance " + instanceId);
    }

    private void removeOldInstances() {
        QueryBuilder builder = deleteFrom(SCHEDULER_INSTANCES)
                .where(SCHEDULER_INSTANCES.HEARTBEAT_TIME.lt(Instant.now().minus(instancesExpiration)));

        var queryAndParams = builder.queryAndParams(path);
        ydbClient.executeQuery(queryAndParams, "Failed to delete very old instances");
    }

    public static RepositoryBuilder builder(TableClient tableClient, YdbPath ydbPath) {
        return new RepositoryBuilder(tableClient, ydbPath);
    }

    public static class RepositoryBuilder {
        private final TableClient tableClient;
        private Duration heartbeatExpiration;
        private Duration instancesExpiration = DEFAULT_INSTANCES_EXPIRATION;
        private int leaderVotingLowerBound;
        private YdbPath ydbPath;

        private RepositoryBuilder(TableClient tableClient, YdbPath ydbPath) {
            this.tableClient = tableClient;
            this.ydbPath = ydbPath;
        }

        public RepositoryBuilder withHeartbeatExpiration(Duration heartbeatExpiration) {
            this.heartbeatExpiration = heartbeatExpiration;
            return this;
        }

        public RepositoryBuilder withInstancesExpiration(Duration instancesExpiration) {
            this.instancesExpiration = instancesExpiration;
            return this;
        }

        public RepositoryBuilder withLeaderVotingLowerBound(int leaderVotingLowerBound) {
            this.leaderVotingLowerBound = leaderVotingLowerBound;
            return this;
        }

        public YdbScheduleInstancesRepository build() {
            return new YdbScheduleInstancesRepository(tableClient, ydbPath, heartbeatExpiration,
                    leaderVotingLowerBound, instancesExpiration);
        }
    }

}
