package ru.yandex.direct.hourglass.ydb.storage;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

import ru.yandex.direct.hourglass.InstanceId;
import ru.yandex.direct.hourglass.storage.Job;
import ru.yandex.direct.hourglass.storage.JobStatus;
import ru.yandex.direct.hourglass.storage.PrimaryId;
import ru.yandex.direct.hourglass.storage.TaskId;
import ru.yandex.direct.hourglass.storage.implementations.Find;
import ru.yandex.direct.ydb.builder.predicate.Predicate;

import static ru.yandex.direct.hourglass.ydb.storage.ScheduledTasksStatus.DELETED;
import static ru.yandex.direct.hourglass.ydb.storage.ScheduledTasksStatus.NEW;
import static ru.yandex.direct.hourglass.ydb.storage.ScheduledTasksStatus.PAUSED;
import static ru.yandex.direct.hourglass.ydb.storage.ScheduledTasksStatus.RUNNING;
import static ru.yandex.direct.hourglass.ydb.storage.Tables.SCHEDULED_TASKS;
import static ru.yandex.direct.hourglass.ydb.storage.YdbStorageImpl.UNIVERSAL_VERSION;

public class YdbFindImpl implements Find {
    private final YdbStorageImpl storage;
    private List<PrimaryId> primaryIdsToOptimize = new ArrayList<>();
    private JobStatus jobStatus = null;
    private Boolean needReschedule = null;
    private TaskId taskId = null;
    private final String version;
    private final InstanceId instanceId;
    private final Duration taskHeartbeatExpiration;
    private final YdbResultToJobMapper ydbResultToJobMapper = new YdbResultToJobMapper();

    private Predicate predicate;

    YdbFindImpl(YdbStorageImpl storage, String version, InstanceId instanceId, Duration taskHeartbeatExpiration) {
        this.storage = storage;
        this.version = version;
        this.instanceId = instanceId;
        this.taskHeartbeatExpiration = taskHeartbeatExpiration;
    }

    private Predicate addVersionCondition(Predicate startPredicate) {
        if (!UNIVERSAL_VERSION.equals(version)) {
            startPredicate = startPredicate.and(SCHEDULED_TASKS.VERSION.eq(version));
        }
        return startPredicate;
    }

    private Predicate statusToCondition(JobStatus jobStatus) {
        switch (jobStatus) {
            case ARCHIVED:
                return SCHEDULED_TASKS.STATUS.eq(DELETED.getDbString());

            case READY:
                return addVersionCondition(SCHEDULED_TASKS.STATUS.eq(NEW.getDbString()).and(SCHEDULED_TASKS.INSTANCE_ID.isNull()));

            case EXPIRED:
                return SCHEDULED_TASKS.INSTANCE_ID.isNotNull().and(SCHEDULED_TASKS.STATUS.eq(RUNNING.getDbString())).and(SCHEDULED_TASKS.HEARTBEAT_TIME.le(
                        Instant.now().minus(taskHeartbeatExpiration)));

            case STOPPED:
                return SCHEDULED_TASKS.STATUS.eq(PAUSED.getDbString());

            case LOCKED:
                return SCHEDULED_TASKS.STATUS.eq(RUNNING.getDbString()).and(SCHEDULED_TASKS.INSTANCE_ID.eq(instanceId.toString()));

            default:
                throw new IllegalStateException("Unknown  state " + jobStatus);
        }
    }

    Find wherePrimaryIdIn(Collection<PrimaryId> primaryIds, boolean needOptimization) {
        if (primaryIds.size() == 1) {
            addPredicate(SCHEDULED_TASKS.ID.eq(primaryIds.iterator().next().toString()));
        } else {
            if (!needOptimization) {
                addPredicate(SCHEDULED_TASKS.ID.in(primaryIds.stream().map(PrimaryId::toString).collect(Collectors.toList())));
            } else {
                primaryIdsToOptimize.addAll(primaryIds);
            }
        }
        return this;
    }

    @Override
    public Find wherePrimaryIdIn(Collection<PrimaryId> primaryIds) {
        return wherePrimaryIdIn(primaryIds, true);
    }

    @Override
    public Find whereJobStatus(JobStatus jobStatus) {
        addPredicate(statusToCondition(jobStatus));
        return this;
    }

    @Override
    public Find whereNextRunLeNow() {
        addPredicate(SCHEDULED_TASKS.NEXT_RUN.le(Instant.now()));
        return this;
    }

    @Override
    public Find whereTaskId(TaskId taskId) {
        addPredicate(SCHEDULED_TASKS.NAME.eq(taskId.name()).and(SCHEDULED_TASKS.PARAMS.eq(taskId.param())));
        return this;
    }

    private void addPredicate(Predicate addingPredicate) {
        if (predicate == null) {
            predicate = addingPredicate;
        } else {
            predicate = predicate.and(addingPredicate);
        }
    }

    JobStatus getJobStatus() {
        return jobStatus;
    }

    Boolean getNeedReschedule() {
        return needReschedule;
    }

    TaskId getTaskId() {
        return taskId;
    }

    Predicate getPredicate() {
        return predicate;
    }

    @Override
    public Find whereNeedReschedule(boolean b) {
        addPredicate(SCHEDULED_TASKS.NEED_RESCHEDULE.eq(b));
        return this;
    }

    @Override
    public Collection<Job> findJobs(int limit) {
        return storage.find(predicate, primaryIdsToOptimize, limit, ydbResultToJobMapper);
    }

    @Override
    public Collection<PrimaryId> findPrimaryIds(int limit) {
        return storage.find(predicate, primaryIdsToOptimize, limit,
                reader -> new YdbPrimaryId(reader.getValueReader(SCHEDULED_TASKS.ID).getUtf8()), SCHEDULED_TASKS.ID);
    }


    @Override
    public Collection<PrimaryId> findPrimaryIds() {
        return findPrimaryIds(Integer.MAX_VALUE);
    }

    @Override
    public Collection<Job> findJobs() {
        return findJobs(Integer.MAX_VALUE);
    }
}
