package ru.yandex.direct.hourglass.ydb.storage;

import java.time.Instant;
import java.time.ZoneOffset;
import java.util.function.Function;

import ru.yandex.direct.hourglass.implementations.InstanceIdImpl;
import ru.yandex.direct.hourglass.implementations.TaskProcessingResultImpl;
import ru.yandex.direct.hourglass.storage.Job;
import ru.yandex.direct.hourglass.storage.JobStatus;
import ru.yandex.direct.hourglass.storage.TaskId;
import ru.yandex.direct.hourglass.storage.implementations.JobImpl;
import ru.yandex.direct.hourglass.storage.implementations.TaskIdImpl;
import ru.yandex.direct.ydb.client.ResultSetReaderWrapped;
import ru.yandex.direct.ydb.column.Column;

import static ru.yandex.direct.hourglass.storage.JobStatus.ARCHIVED;
import static ru.yandex.direct.hourglass.storage.JobStatus.LOCKED;
import static ru.yandex.direct.hourglass.storage.JobStatus.READY;
import static ru.yandex.direct.hourglass.storage.JobStatus.STOPPED;
import static ru.yandex.direct.hourglass.ydb.storage.Tables.SCHEDULED_TASKS;

public class YdbResultToJobMapper implements Function<ResultSetReaderWrapped, Job> {
    @Override
    public Job apply(ResultSetReaderWrapped resultSetReader) {
        TaskId taskId = new TaskIdImpl(resultSetReader.getValueReader(SCHEDULED_TASKS.NAME).getUtf8(),
                resultSetReader.getValueReader(SCHEDULED_TASKS.PARAMS).getUtf8());
        String status = resultSetReader.getValueReader(SCHEDULED_TASKS.STATUS).getUtf8();
        String jobInstanceId = getOptionalUtf8(resultSetReader, SCHEDULED_TASKS.INSTANCE_ID);
        return new JobImpl(new YdbPrimaryId(resultSetReader.getValueReader(SCHEDULED_TASKS.ID).getUtf8()),
                taskId,
                resultSetReader.getValueReader(SCHEDULED_TASKS.SCHEDULE_HASH).getUtf8(),
                TaskProcessingResultImpl.builder()
                        .withLastFinishTime(getOptionalInstant(resultSetReader, SCHEDULED_TASKS.LAST_FINISH_TIME))
                        .withLastStartTime(getOptionalInstant(resultSetReader, SCHEDULED_TASKS.LAST_START_TIME))
                        .build(),
                getOptionalInstant(resultSetReader, SCHEDULED_TASKS.NEXT_RUN),
                parseJobStatus(status, jobInstanceId),
                resultSetReader.getValueReader(SCHEDULED_TASKS.NEED_RESCHEDULE).getBool(),
                new InstanceIdImpl(jobInstanceId),
                getOptionalUtf8(resultSetReader, SCHEDULED_TASKS.META));
    }

    private Instant getOptionalInstant(ResultSetReaderWrapped resultSetReaderWrapped,
                                       Column<Instant> column) {
        var valueReader = resultSetReaderWrapped.getValueReader(column);
        return valueReader.isOptionalItemPresent() ? valueReader.getDatetime().toInstant(ZoneOffset.UTC) : null;
    }

    private String getOptionalUtf8(ResultSetReaderWrapped resultSetReaderWrapped,
                                   Column<String> column) {
        var valueReader = resultSetReaderWrapped.getValueReader(column);
        return valueReader.isOptionalItemPresent() ? valueReader.getUtf8() : null;
    }

    private JobStatus parseJobStatus(String status, String jobInstanceId) {

        if (status.equals("New") && jobInstanceId == null) {
            return READY;
        } else if (status.equals("Paused")) {
            return STOPPED;
        } else if (status.equals("Deleted")) {
            return ARCHIVED;
        } else if (jobInstanceId != null) {
            return LOCKED;
        }

        return null;
    }
}
