package ru.yandex.direct.hourglass.ydb.storage;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Function;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.yandex.ydb.table.SessionRetryContext;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.transaction.TxControl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.hourglass.implementations.updateschedule.ScheduleRecord;
import ru.yandex.direct.hourglass.storage.Job;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.ydb.YdbPath;
import ru.yandex.direct.ydb.builder.querybuilder.QueryBuilder;
import ru.yandex.direct.ydb.client.ResultSetReaderWrapped;
import ru.yandex.direct.ydb.client.YdbClient;
import ru.yandex.direct.ydb.exceptions.YdbExecutionQueryException;

import static com.yandex.ydb.core.StatusCode.PRECONDITION_FAILED;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.hourglass.storage.JobStatus.ARCHIVED;
import static ru.yandex.direct.hourglass.ydb.storage.Tables.SCHEDULED_TASKS;
import static ru.yandex.direct.hourglass.ydb.storage.Tables.SCHEDULER_INSTANCES;
import static ru.yandex.direct.ydb.builder.querybuilder.InsertBuilder.insertInto;
import static ru.yandex.direct.ydb.builder.querybuilder.SelectBuilder.select;
import static ru.yandex.direct.ydb.builder.querybuilder.SelectBuilder.selectDistinct;
import static ru.yandex.direct.ydb.builder.querybuilder.UpdateBuilder.set;
import static ru.yandex.direct.ydb.builder.querybuilder.UpdateBuilder.update;
import static ru.yandex.direct.ydb.table.temptable.TempTable.tempTable;

class YdbScheduleUpdater {
    private static final Logger logger = LoggerFactory.getLogger(YdbScheduleUpdater.class);
    private static final Duration QUERY_TIMEOUT = Duration.ofMinutes(1);

    private final SessionRetryContext sessionRetryContext;
    private final YdbPath path;
    private final String version;
    private final IdHashGenerator idHashGenerator;
    private ExecutorService executor;
    private final YdbClient ydbClient;
    private final Function<ResultSetReaderWrapped, Job> readerToJobMapper = new YdbResultToJobMapper();

    YdbScheduleUpdater(TableClient tableClient, YdbPath path, String version, IdHashGenerator idHashGenerator) {
        ThreadFactory threadFactory =
                new ThreadFactoryBuilder().setNameFormat("schedule-ydb-updater-%02d").setDaemon(true).build();
        this.executor = Executors.newFixedThreadPool(5, threadFactory);
        this.sessionRetryContext = SessionRetryContext.create(tableClient)
                .maxRetries(5)
                .retryNotFound(false)
                .executor(executor)
                .build();
        this.path = path;
        this.version = version;
        this.ydbClient = new YdbClient(sessionRetryContext, QUERY_TIMEOUT);
        this.idHashGenerator = idHashGenerator;
    }

    void setNewSchedule(Collection<ScheduleRecord> newScheduledRecords) {
        if (allTasksHasStorageVersion()) {
            return;
        }
        logger.info("Start changing schedule version to {}", version);
        var currentSchedules = getCurrentSchedule();
        Map<String, ScheduleRecord> newScheduleIds =
                newScheduledRecords.stream().collect(toMap(scheduledRecord -> getId(scheduledRecord.getName(),
                        scheduledRecord.getParam()), scheduledRecord -> scheduledRecord));

        Map<String, Job> oldScheduleIds =
                currentSchedules.stream().collect(toMap(job -> ((YdbPrimaryId) job.primaryId()).getId(), job -> job));
        List<ScheduleRecordWithId> newSchedules = new ArrayList<>();
        List<String> notChangedTasks = new ArrayList<>();
        List<ScheduleRecordWithId> rescheduledTasks = new ArrayList<>();
        List<ScheduleRecordWithId> unarchivedTasks = new ArrayList<>();
        List<String> archivedTasks = new ArrayList<>();

        Set<String> allIds = new HashSet<>(newScheduleIds.keySet());
        allIds.addAll(oldScheduleIds.keySet());

        for (var id : allIds) {
            var newScheduleRecord = newScheduleIds.get(id);
            var oldScheduleJob = oldScheduleIds.get(id);
            if (Objects.nonNull(newScheduleRecord) && Objects.isNull(oldScheduleJob)) {
                newSchedules.add(new ScheduleRecordWithId(newScheduleIds.get(id), id));
            }
            if (Objects.nonNull(newScheduleRecord) && Objects.nonNull(oldScheduleJob)) {
                if (oldScheduleJob.jobStatus().equals(ARCHIVED)) {
                    unarchivedTasks.add(new ScheduleRecordWithId(newScheduleIds.get(id), id));
                } else if (!newScheduleRecord.getScheduleHashSum().equals(oldScheduleIds.get(id).getScheduleHash())) {
                    rescheduledTasks.add(new ScheduleRecordWithId(newScheduleIds.get(id), id));
                } else {
                    notChangedTasks.add(id);
                }

                // update meta ?
            }
            if (!newScheduleIds.containsKey(id) && oldScheduleIds.containsKey(id)) {
                archivedTasks.add(id);
            }
        }
        var futures = new ArrayList<CompletableFuture<Void>>();

        futures.add(rescheduleTasks(rescheduledTasks));
        futures.add(unarchiveTasks(unarchivedTasks));
        futures.add(addNewTasks(newSchedules));
        futures.add(archiveTasks(archivedTasks));
        futures.add(changeVersion(notChangedTasks));
        try {
            CompletableFuture.allOf(futures.toArray(CompletableFuture[]::new)).get(QUERY_TIMEOUT.getSeconds(),
                    TimeUnit.SECONDS);
        } catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(ex);
        } catch (ExecutionException | TimeoutException ex) {
            throw new YdbExecutionQueryException(ex);
        }
    }

    CompletableFuture<Void> addNewTasks(List<ScheduleRecordWithId> newTasks) {
        if (newTasks.isEmpty()) {
            return CompletableFuture.completedFuture(null);
        }


        var tempTable = tempTable(SCHEDULED_TASKS.ID, SCHEDULED_TASKS.NAME,
                SCHEDULED_TASKS.PARAMS, SCHEDULED_TASKS.SCHEDULE_HASH, SCHEDULED_TASKS.NEED_RESCHEDULE,
                SCHEDULED_TASKS.STATUS, SCHEDULED_TASKS.VERSION, SCHEDULED_TASKS.META).createValues();

        for (var newTask : newTasks) {
            var scheduleRecord = newTask.scheduleRecord;
            tempTable.fill(newTask.id, scheduleRecord.getName(), scheduleRecord.getParam(),
                    scheduleRecord.getScheduleHashSum(), true, "New", version, scheduleRecord.getMeta());
        }

        QueryBuilder builder = insertInto(SCHEDULED_TASKS)
                .selectAll()
                .from(tempTable);

        var queryAndParam = builder.queryAndParams(path);
        TxControl txControl = TxControl.serializableRw().setCommitTx(true);
        return sessionRetryContext.supplyResult(session -> session.executeDataQuery(queryAndParam.getQuery(),
                txControl, queryAndParam.getParams()))
                .thenAccept(result -> {
                    if (result.getCode().equals(PRECONDITION_FAILED)) {
                        logger.warn("Precondition failed as a result of adding new tasks, another process " +
                                "could do this. Result: {}", result);
                        return;
                    } else if (result.isSuccess()) {
                        logger.info("Successful added {} tasks", newTasks.size());
                        return;
                    }
                    result.expect("Cannot add new " + newTasks.size() + " tasks");
                });
    }

    private CompletableFuture<Void> rescheduleTasks(List<ScheduleRecordWithId> tasksToRescheduleWithId) {
        CompletableFuture<Void> startFuture = CompletableFuture.completedFuture(null);
        for (var taskToRescheduleWithId : tasksToRescheduleWithId) {
            var taskToReschedule = taskToRescheduleWithId.scheduleRecord;
            QueryBuilder builder = update(SCHEDULED_TASKS,
                    set(SCHEDULED_TASKS.NEED_RESCHEDULE, true)
                            .set(SCHEDULED_TASKS.VERSION, version)
                            .set(SCHEDULED_TASKS.SCHEDULE_HASH, taskToReschedule.getScheduleHashSum())
                            .set(SCHEDULED_TASKS.META, taskToReschedule.getMeta()))
                    .where(SCHEDULED_TASKS.VERSION.neq(version)
                            .and(SCHEDULED_TASKS.ID.eq(taskToRescheduleWithId.id)));
            var queryAndParams = builder.queryAndParams(path);
            startFuture = startFuture.thenCompose(start -> ydbClient.executeQueryAsync(queryAndParams, "cannot " +
                    "reschedule task " + taskToRescheduleWithId.id).thenAccept(unused -> {
            }));
        }
        return startFuture;
    }

    private CompletableFuture<Void> unarchiveTasks(List<ScheduleRecordWithId> tasksToRescheduleWithId) {
        CompletableFuture<Void> startFuture = CompletableFuture.completedFuture(null);
        for (var taskToRescheduleWithId : tasksToRescheduleWithId) {
            var taskToReschedule = taskToRescheduleWithId.scheduleRecord;
            QueryBuilder builder = update(SCHEDULED_TASKS,
                    set(SCHEDULED_TASKS.NEED_RESCHEDULE, true)
                            .set(SCHEDULED_TASKS.VERSION, version)
                            .set(SCHEDULED_TASKS.SCHEDULE_HASH, taskToReschedule.getScheduleHashSum())
                            .set(SCHEDULED_TASKS.META, taskToReschedule.getMeta())
                            .set(SCHEDULED_TASKS.STATUS, "New")
                            .setNull(SCHEDULED_TASKS.INSTANCE_ID))
                    .where(SCHEDULED_TASKS.VERSION.neq(version)
                            .and(SCHEDULED_TASKS.STATUS.eq("Deleted")
                                    .and(SCHEDULED_TASKS.ID.eq(taskToRescheduleWithId.id))));

            var queryAndParams = builder.queryAndParams(path);
            ydbClient.executeQueryAsync(queryAndParams, "cannot unarchive tasks " + taskToRescheduleWithId.id);
            startFuture = startFuture.thenCompose(start -> ydbClient.executeQueryAsync(queryAndParams,
                    "cannot unarchive tasks " + taskToRescheduleWithId.id)
                    .thenAccept(unused -> {
                    }));

        }
        return startFuture;
    }

    private CompletableFuture<Void> archiveTasks(List<String> idsToArchive) {
        QueryBuilder builder = update(SCHEDULED_TASKS, set(SCHEDULED_TASKS.STATUS, "Deleted")
                .set(SCHEDULED_TASKS.VERSION, version))
                .where(SCHEDULED_TASKS.ID.in(idsToArchive).and(SCHEDULED_TASKS.VERSION.neq(version)));
        var queryAndParams = builder.queryAndParams(path);
        return ydbClient.executeQueryAsync(queryAndParams, "cannot archive tasks").thenAccept(unused -> {
        });
    }

    private CompletableFuture<Void> changeVersion(List<String> idsToChangeVersion) {
        QueryBuilder builder = update(SCHEDULED_TASKS, set(SCHEDULED_TASKS.VERSION, version))
                .where(SCHEDULED_TASKS.ID.in(idsToChangeVersion)
                        .and(SCHEDULED_TASKS.VERSION.neq(version)));

        var queryAndParams = builder.queryAndParams(path);
        return ydbClient.executeQueryAsync(queryAndParams,
                "Cannot change version on " + version).thenAccept(unused -> {
        });
    }

    private String getId(String name, String param) {
        return idHashGenerator.getHash(name + "_" + param);
    }

    boolean allTasksHasStorageVersion() {
        var queryBuilder = selectDistinct(SCHEDULED_TASKS.VERSION)
                .from(SCHEDULED_TASKS);

        var queryAndParams = queryBuilder.queryAndParams(path);
        var resultReader = ydbClient.executeQuery(queryAndParams, "expected success result", true).getResultSet(0);

        if (resultReader.getRowCount() == 1) {
            resultReader.next();
            return version.equals(resultReader.getValueReader(SCHEDULER_INSTANCES.VERSION).getUtf8());
        }
        return false;
    }

    private Set<Job> getCurrentSchedule() {
        var jobs = new HashSet<Job>();
        String maxId = "";
        boolean queryTruncated = true;
        while (queryTruncated) {
            var queryAndParams =
                    select()
                            .from(SCHEDULED_TASKS)
                            .where(SCHEDULED_TASKS.ID.gt(maxId))
                            .queryAndParams(path);
            var resultSet =
                    ydbClient.executeQuery(queryAndParams, "Failed to get current schedule", true).getResultSet(0);
            queryTruncated = resultSet.isTruncated();
            while (resultSet.next()) {
                jobs.add(readerToJobMapper.apply(resultSet));
                maxId = resultSet.getValueReader(SCHEDULED_TASKS.ID).getUtf8();
            }
        }
        return jobs;
    }

    static class ScheduleRecordWithId {
        private ScheduleRecord scheduleRecord;
        private String id;

        ScheduleRecordWithId(ScheduleRecord scheduleRecord, String id) {
            this.scheduleRecord = scheduleRecord;
            this.id = id;
        }

    }

    void close() {
        this.executor.shutdown();
    }
}
