package ru.yandex.direct.hourglass.ydb.storage;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.function.Function;

import javax.annotation.Nullable;

import com.yandex.ydb.table.SessionRetryContext;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.values.PrimitiveType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.hourglass.HourglassProperties;
import ru.yandex.direct.hourglass.InstanceId;
import ru.yandex.direct.hourglass.implementations.updateschedule.ScheduleRecord;
import ru.yandex.direct.hourglass.storage.PrimaryId;
import ru.yandex.direct.hourglass.storage.Storage;
import ru.yandex.direct.hourglass.storage.Update;
import ru.yandex.direct.hourglass.storage.implementations.Find;
import ru.yandex.direct.ydb.YdbPath;
import ru.yandex.direct.ydb.builder.predicate.Predicate;
import ru.yandex.direct.ydb.builder.querybuilder.QueryBuilder;
import ru.yandex.direct.ydb.builder.querybuilder.UpdateBuilder;
import ru.yandex.direct.ydb.client.DataQueryResultWrapper;
import ru.yandex.direct.ydb.client.ResultSetReaderWrapped;
import ru.yandex.direct.ydb.client.YdbClient;
import ru.yandex.direct.ydb.client.YdbSessionProperties;
import ru.yandex.direct.ydb.column.Column;
import ru.yandex.direct.ydb.column.TempColumn;
import ru.yandex.direct.ydb.table.temptable.TempTable1;
import ru.yandex.direct.ydb.table.temptable.TempTableDescription;

import static ru.yandex.direct.hourglass.ydb.storage.Tables.SCHEDULED_TASKS;
import static ru.yandex.direct.hourglass.ydb.storage.YdbStorageImpl.Ids.IDS;
import static ru.yandex.direct.ydb.builder.predicate.Predicate.not;
import static ru.yandex.direct.ydb.builder.querybuilder.JoinBuilder.JoinStatement.on;
import static ru.yandex.direct.ydb.builder.querybuilder.SelectBuilder.select;


public class YdbStorageImpl implements Storage, AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(YdbStorageImpl.class);
    static final String UNIVERSAL_VERSION = "";
    private static final Duration QUERY_TIMEOUT = Duration.ofMinutes(1);

    private final TableClient client;
    private final YdbPath path;
    private final HourglassProperties hourglassProperties;
    private final InstanceId instanceId;
    private final String version;
    private final YdbScheduleUpdater scheduleUpdater;
    private final YdbClient ydbClient;

    public YdbStorageImpl(TableClient tableClient, YdbPath path, YdbSessionProperties hourglassYdbProperties,
                          HourglassProperties hourglassProperties,
                          InstanceId instanceId, String version, IdHashGenerator idHashGenerator) {
        this.client = tableClient;
        var sessionRetryContext = SessionRetryContext.create(client)
                .maxRetries(hourglassYdbProperties.getMaxQueryRetries())
                .retryNotFound(hourglassYdbProperties.isRetryNotFound())
                .build();
        this.path = path;
        this.hourglassProperties = hourglassProperties;
        this.instanceId = instanceId;
        this.version = version;
        this.scheduleUpdater = new YdbScheduleUpdater(client, path, version, idHashGenerator);
        this.ydbClient = new YdbClient(sessionRetryContext, QUERY_TIMEOUT);
    }

    public YdbStorageImpl(TableClient tableClient, YdbPath path, YdbSessionProperties hourglassYdbProperties,
                          HourglassProperties hourglassProperties,
                          InstanceId instanceId, IdHashGenerator idHashGenerator) {
        this(tableClient, path, hourglassYdbProperties, hourglassProperties, instanceId, UNIVERSAL_VERSION,
                idHashGenerator);
    }

    @Override
    public void setNewSchedule(Collection<ScheduleRecord> newScheduledRecords) {
        scheduleUpdater.setNewSchedule(newScheduledRecords);
    }

    @Override
    public Find find() {
        return new YdbFindImpl(this, version, instanceId, hourglassProperties.getTaskHeartbeatExpiration());
    }

    @Override
    public Update update() {
        return new YdbUpdateImpl(this, version, instanceId, hourglassProperties.getTaskHeartbeatExpiration());
    }


    <T> Collection<T> find(Predicate findPredicate, List<PrimaryId> primaryIds, int limit,
                           Function<ResultSetReaderWrapped, T> mapper, Column... columns) {
        if (!primaryIds.isEmpty()) {
            return findInternalWithPrimaryIdsFilter(primaryIds, findPredicate, limit, mapper, columns);
        } else {
            return findInternal(findPredicate, limit, mapper, columns);
        }
    }

    private <T> Collection<T> findInternal(@Nullable Predicate findPredicate, int limit,
                                           Function<ResultSetReaderWrapped, T> mapper,
                                           Column... columns) {
        var selectBuilder = columns.length == 0 ? select() : select(columns);
        return executeWithPaging(whereFilter -> selectBuilder
                .from(SCHEDULED_TASKS)
                .where(whereFilter)
                .limit(limit), findPredicate, mapper);
    }

    private <T> Collection<T> findInternalWithPrimaryIdsFilter(List<PrimaryId> primaryIds,
                                                               @Nullable Predicate findPredicate,
                                                               int limit,
                                                               Function<ResultSetReaderWrapped, T> mapper,
                                                               Column... columns) {
        var selectBuilder = columns.length == 0 ? select() : select(columns);
        var primaryIdTable = IDS.createValues();
        for (var primaryId : primaryIds) {
            primaryIdTable.fill(primaryId.toString());
        }
        return executeWithPaging(whereFilter -> selectBuilder
                .from(primaryIdTable)
                .join(SCHEDULED_TASKS, on(IDS.id, SCHEDULED_TASKS.ID))
                .where(whereFilter)
                .limit(limit), findPredicate, mapper);
    }

    private <T> Collection<T> executeWithPaging(Function<Predicate, QueryBuilder> gueryBuilderFromPredicate,
                                                @Nullable Predicate predicate,
                                                Function<ResultSetReaderWrapped, T> mapper) {
        List<T> resultCollection = new ArrayList<>();
        boolean queryTruncated = true;
        String maxId = "";
        while (queryTruncated) {
            Predicate predicateWithPrimaryIdPaging;
            // ВАЖНО: используем отрицание, чтобы фильтр (id > maxId) не стал предикатом доступа (DIRECT-162076)
            if (Objects.isNull(predicate)) {
                predicateWithPrimaryIdPaging = not(SCHEDULED_TASKS.ID.le(maxId));
            } else {
                predicateWithPrimaryIdPaging = predicate.and(not(SCHEDULED_TASKS.ID.le(maxId)));
            }
            QueryBuilder queryBuilder = gueryBuilderFromPredicate.apply(predicateWithPrimaryIdPaging);

            ResultSetReaderWrapped reader = executeQuery(queryBuilder).getResultSet(0);
            queryTruncated = reader.isTruncated();
            while (reader.next()) {
                resultCollection.add(mapper.apply(reader));
                maxId = reader.getValueReader(SCHEDULED_TASKS.ID).getUtf8();
            }
        }
        return resultCollection;
    }


    int executeUpdate(YdbUpdateImpl update) {
        var where = update.getFindCondition();
        var setStatement = update.getSetStatement();
        if (setStatement == null) {
            return 0;
        }
        var wherePredicate = where.getPredicate();
        QueryBuilder builder;
        if (wherePredicate != null) {
            builder = UpdateBuilder.update(SCHEDULED_TASKS, setStatement).where(wherePredicate);
        } else {
            builder = UpdateBuilder.update(SCHEDULED_TASKS, setStatement);
        }
        executeQuery(builder);
        return 0;
    }

    private DataQueryResultWrapper executeQuery(QueryBuilder builder) {
        var queryAndParams = builder.queryAndParams(path);
        return ydbClient.executeQuery(queryAndParams, true);
    }

    @Override
    public void close() {
        scheduleUpdater.close();
        client.close();
        logger.info("Table client closed");
    }

    static class Ids extends TempTable1<String> {

        static final Ids IDS = new Ids();
        TempColumn<String> id;

        Ids(TempTableDescription tableDescription, TempColumn<String> column) {
            super(tableDescription, column);
            this.id = column;
        }

        Ids(TempTableDescription tableDescription) {
            this(tableDescription, TempColumn.tempCol(tableDescription, "id", PrimitiveType.utf8()));
        }

        Ids() {
            this(new TempTableDescription("ids"));
        }
    }
}
